/**
 * Interface for Applications (Header)
 * 
 * Copyright 2013 Fabian Schrodt, FSchrodt@gmx.de
 * 
 * This file is part of RNNPBlib.
 * 
 * RNNPBlib is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License version 3 as published by the Free Software Foundation.
 * 
 * RNNPBlib is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with RNNPBlib. If not, see http://www.gnu.org/licenses/.
 */

#pragma once

#include "RNNPB_Definitions.h"

#include "RNNPB_Helper.h"
#include "RNNPB_TrainingData.h"

#include <vector>
using std::vector;

/*
 * Classes which use the RNNPB-Trainer must derive from this interface.
 */
class RNNPB_ApplicationInterface
{
	friend class RNNPB_CVTrainer;
	friend class RNNPB_CVOfflineTrainer;

	/*
	 * May be implemented.
	 */
	virtual void action(RNNPB_Vector output);

	/*
	 * MUST be implemented!
	 * Gets input-vector for NN dependent on iteration
	 * */
	virtual void get_nn_input(RNNPB_Vector* return_value) = 0;

	/*
	 * MUST be implemented!
	 * Gets target for the output of NN dependent on iteration
	 * switches feedback_method()
	 * */
	virtual void get_nn_target(RNNPB_Vector* return_value) = 0;

	RNNPB_Vector iteration;

	RNNPB_Vector min_iteration;
	RNNPB_Vector max_iteration;

	/*
	 * Gets the maximum number of iterations necessary to cover the input space of the neural network
	 */
	unsigned int get_max_iteration();

	unsigned int nr_of_iterations;
	unsigned int nr_of_overall_iterations;
	unsigned int iterations_done;

	bool random_sampling;

	unsigned int feedback_method;

	int phase;

protected:

	/*
	 * Gets the selected phase (see documentation).
	 *
	 * Predefined phases:
	 * 	       -1: Undefined
	 * 		0: Offline-data will be generated
	 * 		1: The network is being trained
	 * 		2: The network is being tested
	 *
	 * Phases > 2 can be set manually (set_phase / reset_phase):
	 *
	 * */
	int get_phase();

	void set_phase(int set_phase);

	void reset_phase();

	void iterate();

	/*
	 * Feedback-method will be set by trainer, so this is equivalent to the selected-behaviour.
	 * You can use this to distinguish different responses of the application.
	 */
	unsigned int get_feedback_method();

	/*
	 * Gets the number of iterations done in the current epoch.
	 */
	unsigned int get_iteration();

	/*
	 * May be implemented to initialize some stuff of the application before an epoch begins.
	 * */
	virtual void initEpoch();

	/*
	 * May be implemented to stop deinitialize some stuff of the application after an epoch stops.
	 */
	virtual void stopEpoch();

public:

	/*
	 * This function can be used to cover the input space of a neural network (see documentation).
	 */
	RNNPB_Vector get_app_input();

	RNNPB_Vector get_max_input();

	RNNPB_Vector get_min_input();

	/*
	 * May be implemented to abort an epoch
	 */
	virtual bool abortEpochConditionMet();

	/*
	 * My be implemented to check if an epoch was successful (so it can eventually be repeated)
	 */
	virtual bool epochGoalReached();

	virtual bool abortLearningConditionMet();

	void switch_feedback_method_to(int set_feedback_method);

	RNNPB_ApplicationInterface(unsigned int set_nr_of_iterations, bool set_random_sampling, RNNPB_Vector set_min_iteration, RNNPB_Vector set_max_iteration);
};
