/**
 * Classes and functions for streamable/serial data, see notes below (Header+Implementation)
 * 
 * Copyright 2013 Fabian Schrodt, FSchrodt@gmx.de
 * 
 * This file is part of SimulationManager.
 * 
 * SimulationManager is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License version 3 as published by the Free Software Foundation.
 * 
 * SimulationManager is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with SimulationManager. If not, see http://www.gnu.org/licenses/.
 */

#pragma once

#include <string.h>
#include <stdlib.h>
#include <typeinfo>
#include <iostream>
#include <vector>

using namespace std;

#define DATA_ALIGNMENT 1	//1 or 0. Disabling might both speedup IPC and slow down working on data.

#if DATA_ALIGNMENT == 0
	#define BEGIN_DATA _Pragma("pack(1)")
#else
	#define BEGIN_DATA _Pragma("pack(8)")	//insert word size here (in bytes) to set it manually for IPC between different archs/compiler options.
#endif
#define END_DATA _Pragma("pack()")

/*
 * Implement data that will be handlet by a StreamableDataContainer like this: (mehr oder weniger TODO)
 *
 * Data classes may contain any fundamental, non-referring data types that are coherent in memory.
 * That is, any arrays, but no pointers oder dynamic types like <vector>.
 *
 * BEGIN_DATA		//This disables the byte padding for this struct if DATA_ALIGNMENT is set to 0.
 * struct SimulationSensor : public StreamableDataContainer
 * {
 * 		//any data you would want...
 * 		int intA;
 * 		int intB;
 * 		double doubleA;
 * 		float arrayA[100];
 * 		u_int_16_t shortInt;
 * 		u_int_64_t longInt;
 * };
 * END_DATA		//Reenables padding.
 *
 * You mal also derive a class from StreamableDataContainer to make the type interpretion easier...
* */

/*
 * Prototype for a streamable data-container.
 * All simulation-command bodys/headers and data (sensors/actors) meant to be streamed must derive from this class.
 * ATTENTION: ALWAYS USE THE ABOVE COMPILER DIRECTIVES TO AVOID BYTE PADDING IN SUBCLASSES!
 */
class StreamableDataContainer
{
	friend class PipeServerSimulation;
	friend class PipeClientSimulation;

protected:

	void* dataStream;	//array of size streamSize

	unsigned int streamSize;

public:

	/*
	 * Resize the container losing all data.
	 * */
	void reconfigure(unsigned int set_streamSize)
	{
		if(set_streamSize!=streamSize)	//reallocate
		{
			if(dataStream!=NULL)
			{
				free(dataStream);	//CHECK: does this really delete all data??
				dataStream=NULL;
			}

			if(set_streamSize!=0)
				dataStream = malloc(set_streamSize);

			streamSize=set_streamSize;
		}
	}

	/*
	 * Copies stream data into this container (and adapts its size if necessary)
	 * */
	void setDataStream(const void* stream, const unsigned int set_streamSize)
	{
		reconfigure(set_streamSize);

		if(set_streamSize!=0)
			memcpy(dataStream, (void*) stream, streamSize);
	}
	template <typename T> void setDataStream(const T* datatype)
	{
		reconfigure(sizeof(T));

		memcpy(dataStream, (void*) datatype, streamSize);
	}

	StreamableDataContainer& operator=(StreamableDataContainer& set)
	{
		setDataStream(set.getDataStream(), set.getByteSize());

		return *this;
	}

	//returns pointer to data
	void* getDataStream() const
	{
		return dataStream;
	}

	/*
	 * Returns byte-length of data-stream.
	 * */
	unsigned int getByteSize() const
	{
		return streamSize;
	}

	/*
	 * Interpretes dataStream as a given type.
	 * */
	template <typename T> T* dat()
	{
		return (T*) dataStream;
	}
	template <typename T> T* dat() const
	{
		return (T*) dataStream;
	}


	/*
	 * Deletes data.
	 * */
	void clear()
	{
		reconfigure(0);
	}

	StreamableDataContainer()
	{
		streamSize=0;
		dataStream=NULL;
	}

	StreamableDataContainer(unsigned int set_streamSize)
	{
		streamSize=0;
		dataStream=NULL;

		reconfigure(set_streamSize);
	}

	/*
	 * Copy constructor
	 * */
	StreamableDataContainer(StreamableDataContainer& set)
	{
		setDataStream(set.getDataStream(), set.getByteSize());
	}


	/*
	 * Destructor
	 * Since this class handles the construction of the data, the destructor also frees the memory.
	 * Thus, two StreamableDataContainers cannot share the same memory!
	 * */
	~StreamableDataContainer()
	{
		if(dataStream!=NULL)
			free(dataStream);
	}
};


/*
 * Helper Function to print arrays that implement standard output
 * */
template <class T> void printArray(T* array, unsigned int length)
{
	cout<<"[";
	for(unsigned int i=0;i<length-1;i++)
	{
		cout<<array[i];
		cout<<", ";
	}
	cout<<array[length-1];
	cout<<"]"<<endl;
}

/*
 * Helper Function that gives array size
 * */
template <class T> unsigned int arraySize(T* array)
{
	return sizeof(array) / sizeof(T);
}
