/**
 * Container for Neural Networks (Implementation)
 * 
 * Copyright 2013 Fabian Schrodt, FSchrodt@gmx.de
 * 
 * This file is part of RNNPBlib.
 * 
 * RNNPBlib is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License version 3 as published by the Free Software Foundation.
 * 
 * RNNPBlib is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with RNNPBlib. If not, see http://www.gnu.org/licenses/.
 */

#include "RNNPB_NetworkContainer.h"

void* RNNPB_NetworkContainer::Iterate::func()
{
	#ifdef ENABLE_SUBNET_TREE
	for(unsigned int i=0;i<base->subnets.size();i++)
	{
		base->subnets[i]->iterate.call();
	}
	#endif

	for(unsigned int i=0;i<base->weights.size();i++)
	{
		base->weights[i]->iterate();
	}
	for(unsigned int i=0;i<base->neurons.size();i++)
	{
		base->neurons[i]->iterate();
	}

	#ifdef ENABLE_SUBNET_TREE
	for(unsigned int i=0;i<base->subnets.size();i++)
	{
		base->subnets[i]->iterate.join();
	}
	#endif

	return NULL;
}

void* RNNPB_NetworkContainer::Clear::func()
{
	#ifdef ENABLE_SUBNET_TREE
	for(unsigned int i=0;i<base->subnets.size();i++)
	{
		base->subnets[i]->clear.call();
	}
	#endif

	for(unsigned int i=0;i<base->weights.size();i++)
	{
		base->weights[i]->clear();
	}	
	for(unsigned int i=0;i<base->neurons.size();i++)
	{
		base->neurons[i]->clear();
	}

	#ifdef ENABLE_SUBNET_TREE
	for(unsigned int i=0;i<base->subnets.size();i++)
	{
		base->subnets[i]->clear.join();
	}
	#endif

	return NULL;
}


void* RNNPB_NetworkContainer::CumulateWeightUpdates::func()
{
	//Subnets first
	#ifdef ENABLE_SUBNET_TREE
	//cumulate subnets' weights (threaded)
	for(unsigned int i=0;i<base->subnets.size();i++)
	{
		base->subnets[i]->cumulateWeightUpdates.call();
	}
	for(unsigned int i=0;i<base->subnets.size();i++)
	{
		base->subnets[i]->cumulateWeightUpdates.join();
	}
	#endif

	for(std::vector<RNNPB_Weight*>::iterator it = base->weights.begin() ; it != base->weights.end(); ++it)
	{
		(*it)->cumulateWeightUpdates();
	}

	return NULL;
}

void* RNNPB_NetworkContainer::LearnWeights::func()
{
	#ifdef ENABLE_SUBNET_TREE
	//learn weights of subnets (threaded)
	for(unsigned int i=0;i<base->subnets.size();i++)
	{
		base->subnets[i]->learnWeights.call();
	}
	#endif

	//cout<<"void RNNPB_NetworkContainer::learnWeights()"<<endl;
	for(std::vector<RNNPB_Weight*>::iterator it = base->weights.begin() ; it != base->weights.end(); ++it)
	{
		//if(weights[i]->learningRate>0)
		//	cout<<"WEIGHT: "<<weights[i]->weight<<" -> ";

		(*it)->learn();

		//if(weights[i]->learningRate>0)
		//	cout<<weights[i]->weight<<endl;
	}
	//cout<<endl;
	//usleep(1000000);

	#ifdef ENABLE_SUBNET_TREE
	//learn weights of subnets (threaded)
	for(unsigned int i=0;i<base->subnets.size();i++)
	{
		base->subnets[i]->learnWeights.join();
	}
	#endif

	return NULL;
}


unsigned int RNNPB_NetworkContainer::weightSize()
{
	unsigned int ret=weights.size();

	#ifdef ENABLE_SUBNET_TREE
	for(unsigned int i=0;i<subnets.size();i++)
	{
		ret+=subnets[i]->weightSize();
	}
	#endif

	return ret;
}


void RNNPB_NetworkContainer::registerNeuron(RNNPB_Neuron* add_neuron, int run)
{
	if(run==1)	
		neurons.push_back(add_neuron);
	else
	{
		inverseNeurons.push_back(add_neuron);
	}
}

void RNNPB_NetworkContainer::registerWeight(RNNPB_Weight* add_weight)
{
	weights.push_back(add_weight);
}

void RNNPB_NetworkContainer::registerSubnet(RNNPB_NetworkContainer* add_subnet)
{
	subnets.push_back(add_subnet);
}


void RNNPB_NetworkContainer::printNetwork()
{
	cout<<"Network: "<<neurons.size()<<" neurons, "<<weights.size()<<" weights, "<<subnets.size()<<" subnets.\n";
	#ifdef ENABLE_SUBNET_TREE
		for(unsigned int i=0;i<subnets.size();i++)
		{
			cout<<" -- Subnetwork: ";
			subnets[i]->printNetwork();
		}
	#endif

	/*cout<<"Weights: \n";
	for(unsigned int i=0;i<weights.size();i++)
		cout<<weights[i]->getWeight()<<" ";

	cout<<endl;*/
}

/*
 * This will affect all (sub)nets.
 * */
void RNNPB_NetworkContainer::setWeightsConstant(bool set)
{
	for(unsigned int i=0;i<weights.size();i++)
		weights[i]->constant=set;
	#ifdef ENABLE_SUBNET_TREE
	for(unsigned int i=0;i<subnets.size();i++)
		subnets[i]->setWeightsConstant(set);
	#endif
}

/*
 * (Re)allocates serial memory for all neurons in the network and assigns them in correct order for performance!
 * */
void RNNPB_NetworkContainer::allocateMemory()
{
	memorya = (double*) malloc(neurons.size()*sizeof(double));
	memoryb = (double*) malloc(neurons.size()*sizeof(double));
	memoryc = (double*) malloc(inverseNeurons.size()*sizeof(double));

	cout<<"Buffering feed forward ("<<neurons.size()<<")..."<<endl;
	cout<<"Buffering backpropagation ("<<inverseNeurons.size()<<", "<<neurons.size()-inverseNeurons.size()<<" deltas could not be buffered in read-order)..."<<endl;
	//(bias neurons will not be inverse parsed)

	for(unsigned int i=0;i<neurons.size();i++)
	{
		delete neurons[i]->activation;
		neurons[i]->activation=memorya+i;
		delete neurons[i]->activationDifferential;
		neurons[i]->activationDifferential=memoryb+i;

		/*//pseudo inverse buffering:
		delete neurons[i]->deltaError;
		neurons[i]->deltaError=memoryc+neurons.size()-1-i;*/
	}
	//inverse buffering of deltas:
	for(unsigned int i=0;i<inverseNeurons.size();i++)
	{	
		delete inverseNeurons[i]->deltaError;
		inverseNeurons[i]->deltaError=memoryc+i;
	}
	for(unsigned int i=0;i<neurons.size();i++)
	{
		neurons[i]->clear();
	}
}

void RNNPB_NetworkContainer::deallocateMemory()
{
	delete[] memorya;
	delete[] memoryb;
	delete[] memoryc;
}

RNNPB_NetworkContainer::RNNPB_NetworkContainer(RNNPB_NeuronLayer* set_input, RNNPB_NeuronLayer* set_output, vector <RNNPB_NeuronLayer*>* set_concept_layers)
{
	input=set_input;
	output=set_output;
	concept_layers=set_concept_layers;

	/*
	 * Parse Network(s)
	 * */
	output->parseNetwork(this, 1);
	
	#ifndef ENABLE_SUBNET_TREE
	//allocate deltas starting at concept layers
	if(concept_layers!=NULL)
	{
		for(unsigned int i=0;i<concept_layers->size();i++)
		{
			(*concept_layers)[i]->parseNetwork(this,2);
		}
	}
	#endif

	if(input!=NULL)	//TODO: ENABLE DELTA BUFFER PARSING FOR SUBNETWORKS!!!
		input->parseNetwork(this, 2);

	/*
	 * Initialize serial memory
	 * */
	allocateMemory();
	#ifdef ENABLE_DEBUG
	#ifdef ENABLE_SUBNET_TREE
	cout<<"Warning: Serial memory for subnets is work in progress\n";
	#endif
	#endif
	/*#ifdef ENABLE_SUBNET_TREE
		for(unsigned int i=0;i<subnets.size();i++)
			subnets[i]->allocateMemory();
	#endif <<<< SUBNES ALLOCATE THEIR MEMORY THEIRSELFES...*/

	/*
	 * Initialize thread-function bases
	 * */
	iterate.setBase(this);
	clear.setBase(this);
	cumulateWeightUpdates.setBase(this);
	learnWeights.setBase(this);

	/*
	 * Print Network
	 * */
	cout<<"NETWORK CONTAINER CREATED!\n"<<endl;
	#ifdef ENABLE_DEBUG
	//only parent will print...
	if(concept_layers==NULL)
		printNetwork();	
	#endif
}

RNNPB_NetworkContainer::~RNNPB_NetworkContainer()
{
	deallocateMemory();
	#ifdef ENABLE_SUBNET_TREE
		for(unsigned int i=0;i<subnets.size();i++)
			subnets[i]->deallocateMemory();
	#endif

	//TODO: clean up all stuff??? (neurons, weights, layers)
}
