/**
 * Interface for Plugins (Header+Implementation)
 * 
 * Copyright 2013 Fabian Schrodt, FSchrodt@gmx.de
 * 
 * This file is part of SimulationManager.
 * 
 * SimulationManager is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License version 3 as published by the Free Software Foundation.
 * 
 * SimulationManager is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with SimulationManager. If not, see http://www.gnu.org/licenses/.
 */

#pragma once

#include <pthread.h>
#include <iostream>

class SimulationManager;

class PluginInterface
{
	friend class SimulationManager;

	pthread_t* server_thread;

	pthread_mutex_t running_lock;
	bool running;

	virtual void run() = 0;

	/*
	 * Thread-entry
	 * */
	static void* runThreadEntry(void* context)
	{
		((PluginInterface*) context)->run();
		pthread_mutex_lock(&(((PluginInterface*) context)->running_lock));
		((PluginInterface*) context)->running=false;
		pthread_mutex_unlock(&(((PluginInterface*) context)->running_lock));
		return NULL;
	}

	/*
	 * Will be called if the Manager wants to quit or close the server.
	 * */
	virtual int exit() = 0;

	//TODO: Force definition of plugin name...
	//TODO: Implement some "print" command using plugin name...

public:

	SimulationManager* man;

	bool isRunning()
	{
		bool ret;
		pthread_mutex_lock(&running_lock);
		ret=running;
		pthread_mutex_unlock(&running_lock);
		return ret;
	}

	/*
	 * Constructor
	 * */
	PluginInterface(SimulationManager* set_man)
	{
		pthread_mutex_init(&running_lock, NULL);
		server_thread=new pthread_t();
		man=set_man;
		running=true; //Wird mit true initialisiert, da 1. die konstruktion läuft und 2. anschließend immer direkt run aufgerufen wird.
	}

	~PluginInterface()
	{
		std::cout<<"(Plugin destructor called.)\n";
	}
};

/*
 * Deriving Plugin must define the createServer entry symbol:
 * */
extern "C" PluginInterface* createServer(SimulationManager* man);

