/**
 * Template class for implementing a function as thread (Header+Implementation)
 * 
 * Copyright 2013 Fabian Schrodt, FSchrodt@gmx.de
 * 
 * This file is part of RNNPBlib.
 * 
 * RNNPBlib is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License version 3 as published by the Free Software Foundation.
 * 
 * RNNPBlib is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with RNNPBlib. If not, see http://www.gnu.org/licenses/.
 */

#pragma once

#include "RNNPB_Definitions.h"

#include <pthread.h>
#include <semaphore.h>

//TODO: Limit the number of threads
//#define NR_OF_CPUS 8

/*
 * Any method of a function may be implemented as thread. This template class enables parallel execution by one thread per containing object (see documentation)
 * The function implementation may return a value of type other than void.
 * Warning: Do not use unless the implemented function takes a considerable amount of time, because thread allocation and management takes time too!
*/
template <typename retT, typename contT> class RNNPB_ThreadedFunction
{
	pthread_t		thread;		//single thread, called on creation of the object to avoid reallocation and exceedance of resources.
	pthread_mutex_t		callLock;	//blocks until thread is finished
	pthread_mutex_t		runThread;	//unlocking this triggers a call to the implementation fo the function "func" with the base as context-object

	retT			ret;		//buffer for function returns

	inline static void* func_entry(void* context)
	{
		((RNNPB_ThreadedFunction<retT,contT>*) context)->ret = ((RNNPB_ThreadedFunction<retT,contT>*) context)->functionCaller();
		//pthread_mutex_unlock(&(((RNNPB_ThreadedFunction<retT,contT>*) context)->callLock));

		return NULL;
	}

	void* functionCaller()
	{
		while(1)
		{
			pthread_mutex_lock(&runThread);
			ret = func();
			pthread_mutex_unlock(&callLock);
		}
		return NULL;
	}

protected:
	/*
	 * Base/context class which contains the threaded function implementation
	 */
	contT* base;

public:
	/*
	 * This function will be implemented to contain the parallel task.
	 * It is made public to enable non-threaded calls too.
	 */
	virtual retT func() = 0;

	/*
	 * Gets return value once it is ready.
	 */
	inline retT get()
	{
		pthread_mutex_lock(&callLock);
		retT tmp=ret;
		pthread_mutex_unlock(&callLock);
		return tmp;
	}

	/*
	 * Blocks the caller until the thread finished.
	 */
	inline void join()
	{
		pthread_mutex_lock(&callLock);
		pthread_mutex_unlock(&callLock);
	}

	/*
	 * Calls the implementation of func() as thread.
	 */
	inline void call()
	{		
		pthread_mutex_lock(&callLock);
		pthread_mutex_unlock(&runThread);
	}

	/*
	 * The base/context must be set befor the first call to "call()".
	 * The context can be reset to another class on runtime.
	 */
	void setBase(contT* set_base)
	{
		base = set_base;
	}

	RNNPB_ThreadedFunction()
	{
		pthread_mutex_init(&callLock, NULL);
		pthread_mutex_init(&runThread, NULL);
		pthread_mutex_lock(&runThread);

		pthread_create(&thread, NULL, RNNPB_ThreadedFunction::func_entry, (void*) this);
	}

	//TODO: destructor
};

/*template <typename retT, typename contT> int Thread<retT, contT>::nrOfThreads = 0;
template <typename retT, typename contT> pthread_mutex_t Thread<retT, contT>::globalLock = PTHREAD_MUTEX_INITIALIZER;
template <typename retT, typename contT> pthread_cond_t Thread<retT, contT>::joinAllCond = PTHREAD_COND_INITIALIZER;
template <typename retT, typename contT> pthread_mutex_t Thread<retT, contT>::joinAllLock = PTHREAD_MUTEX_INITIALIZER;*/
