/**
 * Interface for Applications (Implementation)
 * For commentaries about the virtual functions see the header file.
 * 
 * Copyright 2013 Fabian Schrodt, FSchrodt@gmx.de
 * 
 * This file is part of RNNPBlib.
 * 
 * RNNPBlib is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License version 3 as published by the Free Software Foundation.
 * 
 * RNNPBlib is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with RNNPBlib. If not, see http://www.gnu.org/licenses/.
 */

#include "RNNPB_ApplicationInterface.h"

RNNPB_Vector RNNPB_ApplicationInterface::get_max_input()
{
	return max_iteration;
}

RNNPB_Vector RNNPB_ApplicationInterface::get_min_input()
{
	return min_iteration;
}

RNNPB_Vector RNNPB_ApplicationInterface::get_app_input()
{
	return iteration;
}

unsigned int RNNPB_ApplicationInterface::get_iteration()
{
	return iterations_done;
}

unsigned int RNNPB_ApplicationInterface::get_feedback_method()
{
	return feedback_method;
}

unsigned int RNNPB_ApplicationInterface::get_max_iteration()
{
	return nr_of_overall_iterations;
}

int RNNPB_ApplicationInterface::get_phase()
{
	return phase;
}

void RNNPB_ApplicationInterface::reset_phase()
{
	phase=-1;
}

void RNNPB_ApplicationInterface::set_phase(int set_phase)
{
	if(set_phase < 3)
	{
		cout<<"Error: Phase can only be set manually to a value higher than 2. Other values are reserved!\n";
	}
	else
		phase=set_phase;
}

void RNNPB_ApplicationInterface::action(RNNPB_Vector output){}
void RNNPB_ApplicationInterface::initEpoch(){}
void RNNPB_ApplicationInterface::stopEpoch(){}


void RNNPB_ApplicationInterface::switch_feedback_method_to(int set_feedback_method)
{
	feedback_method=set_feedback_method;

	if(!random_sampling)
	{
		iteration[0] = min_iteration[0] - 0.5 * (max_iteration[0] - min_iteration[0]) / nr_of_iterations;
		for(unsigned int i=1;i<iteration.size;i++)
			iteration[i] = min_iteration[i] + 0.5 * (max_iteration[0] - min_iteration[0]) / nr_of_iterations;
	}

	iterations_done=0;
}

void RNNPB_ApplicationInterface::iterate()
{
	#ifdef ENABLE_DEBUG
		if(abortEpochConditionMet())
		{
			cout<<"Warning: Iteration exceeds maximum! You did iterate too often!\n";
		}
	#endif


	if(random_sampling)
	{
		for(unsigned int i=0;i<iteration.size;i++)
			iteration[i] = min_iteration[i] + ((double)rand() / (double)(RAND_MAX)) * (max_iteration[i] - min_iteration[i]);
	}
	else
	{
		//this iterates like nested "for"-loops

		unsigned int i=0;

		//check numeral overflow
		while(i < iteration.size && iteration[i] + (max_iteration[i] - min_iteration[i]) / nr_of_iterations >= max_iteration[i])
		{
			iteration[i] = min_iteration[i] + 0.5 * (max_iteration[i] - min_iteration[i]) / nr_of_iterations ;

			i++;
		}

		if(i<iteration.size)
			iteration[i] += (max_iteration[i] - min_iteration[i]) / nr_of_iterations;
	}

	iterations_done++;
}

/*
 * Default implementation.
 * */
bool RNNPB_ApplicationInterface::abortEpochConditionMet()
{
	if(get_iteration() >= get_max_iteration())
		return true;
	else
		return false;
}

/*
 * Default implementation.
 * */
bool RNNPB_ApplicationInterface::epochGoalReached()
{
	return true;
}

/*
 * Default implementation.
 * */
bool RNNPB_ApplicationInterface::abortLearningConditionMet()
{
	return false;
}

RNNPB_ApplicationInterface::RNNPB_ApplicationInterface
	(
			unsigned int set_nr_of_iterations,	//ATTENTION: If random sampling is set, this sets the number of iterations PER DIMENSION!
			bool set_random_sampling,
			RNNPB_Vector set_min_iteration,
			RNNPB_Vector set_max_iteration
	) : iteration(set_max_iteration.size), min_iteration(set_min_iteration.size), max_iteration(set_max_iteration.size)
{
	min_iteration=set_min_iteration;
	max_iteration=set_max_iteration;

	random_sampling=set_random_sampling;

	nr_of_iterations=set_nr_of_iterations;

	if(random_sampling)
		nr_of_overall_iterations=nr_of_iterations;
	else
		nr_of_overall_iterations=pow((double) nr_of_iterations,(int) min_iteration.size);	//sorry, but microsoft needs explicit typecasting here...

	iterations_done=0;

	feedback_method=0;

	phase=-1;
}
