/**
 * Weighs (Implementation)
 * 
 * Copyright 2013 Fabian Schrodt, FSchrodt@gmx.de
 * 
 * This file is part of RNNPBlib.
 * 
 * RNNPBlib is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License version 3 as published by the Free Software Foundation.
 * 
 * RNNPBlib is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with RNNPBlib. If not, see http://www.gnu.org/licenses/.
 */

#include "RNNPB_Weight.h"
#include "RNNPB_Neuron.h"
#include "RNNPB_NetworkContainer.h"

/*
 * Oberklassen-Implementierungen
 * */

void RNNPB_Weight::subscribeNetwork(RNNPB_NetworkContainer* set_network, int run)
{
	if(parsed==run)
		return;

	parsed=run;
	network=set_network;

	if(run==1)
	{
		from->subscribeNetwork(set_network, run);

		network->registerWeight(this);	//only register in first run
	}
	else
		to->subscribeNetwork(set_network, run);
}

void RNNPB_Weight::iterate()
{
	deltaErrorTimesActivationSumUptodate=false;
}

void RNNPB_Weight::clear()
{
	deltaErrorTimesActivationSumUptodate=false;
	deltaErrorTimesActivationSum=0.0;
}

double RNNPB_Weight::getOutput()	//Vorwärtspropagierung des Inputs. TODO: Output und Weight separieren für delayed weights
{
	return from->getActivation() * getWeight();
}

double RNNPB_Weight::getWeightedDeltaError(RNNPB_Neuron* caller)	//Rückwärtspropagierung des Fehlers
{
	//Rückpropagierung läuft etwas anders für RBFs...
	if(to->isRadial)
	{
		return to->getDeltaError() * (from->getActivation() - getWeight());
	}
	else
	{
		return to->getDeltaError() * getWeight();
	}
}

void RNNPB_Weight::learn()	//Lernen des Gewichts
{
	if(constant)
		return;

	#ifdef ENABLE_DEBUG
	if(!deltaErrorTimesActivationSumUptodate)
	{
		cout<<"Update Error!\n"<<endl;
		//__asm{int 3};
	}
	#endif

	/*
	VGL.:
	RNNPB_Vector old_update = behaviours[behaviour]->concept_vector_update[p];
	RNNPB_Vector new_update = (pb_layers[p]->swapWeights() - behaviours[behaviour]->concept_vector[p]);

	behaviours[behaviour]->concept_vector_update[p] = new_update + behaviours[behaviour]->concept_vector_update[p] * DEFAULT_CV_MOMENTUM;

	behaviours[behaviour]->concept_vector[p] = behaviours[behaviour]->concept_vector[p] + behaviours[behaviour]->concept_vector_update[p] * DEFAULT_CV_LEARNING_RATE;
	*/

	//ohne momentum:
	//weight += deltaErrorTimesActivationSum * learningRate - WEIGHT_DECAY * weight;

	//neues update mit momentum = altes update mit momentum * momentum      +    neues update ohne momentum
	weightUpdate                = weightUpdate * momentum    +    deltaErrorTimesActivationSum / double(from->iteration-1);

	#ifdef ENABLE_SEPARATE_ANNEALING
		//Senke/erhöhe Lernrate so dass das Gewichtsupdate im Korridor verbleibt!
		if(weightUpdate != 0.0)	//avoid division by zero
		{
			if(abs(weightUpdate * learningRate) > maxWeightUpdate)
				learningRate=abs(maxWeightUpdate / weightUpdate);
			else if(abs(weightUpdate * learningRate) < minWeightUpdate)
				learningRate=abs(minWeightUpdate / weightUpdate);
		}
		if(learningRate>1.0)	//maximum learning rate!
			learningRate=1.0;
		/*else if(learningRate<0.0)	//kann eigentlich nicht sein
			learningRate=0.0;*/

		//Senke Korridor ab!
		maxWeightUpdate*=ANNEALING_FACTOR;
		minWeightUpdate*=ANNEALING_FACTOR;

		/*if(maxWeightUpdate <= 0.0)
			maxWeightUpdate = 0.0;
		if(minWeightUpdate <= 0.0)
			minWeightUpdate = 0.0;*/
	#endif

	//neues gewicht = altes gewicht + neues update mit momentum * lernrate
	//cout<<weight;
	weight                         += weightUpdate * learningRate;
	//cout<<" --> "<<weight<<endl;

	#ifdef ENABLE_WEIGHT_DECAY
		weight -= weightDecay * weight;

		weightDecay *= ANNEALING_FACTOR; //annealing of weight-decay
	#endif
	
	#ifdef ENABLE_RESET_WEIGHT
		if(weight>RESET_WEIGHT || weight<-RESET_WEIGHT)
		{
			weight=DEFAULT_INIT_WEIGHT;
			//cout<<"Reset.\n";
		}
	#endif

	#ifdef ENABLE_ANNEALING
		learningRate*=ANNEALING_FACTOR;
	#endif
}

void RNNPB_Weight::cumulateWeightUpdates()	//Aufsummierung der Gewichtsupdates für Batch-Learning
{
	#ifdef ENABLE_DEBUG
	if(!deltaErrorTimesActivationSumUptodate)
	{
	#endif

		if(to->isRadial) //RBF-learning
		{
			deltaErrorTimesActivationSum += to->getDeltaError() * (getWeight() - from->getActivation());
		}
		else
		{
			deltaErrorTimesActivationSum += from->getActivation() * to->getDeltaError();
		}

		deltaErrorTimesActivationSumUptodate=true;

	#ifdef ENABLE_DEBUG
	}
	else
	{
		cout<<"Error: Mehrfachaufruf?!\n";
	}
	#endif
}

double RNNPB_Weight::getWeight()
{
	return weight;
}

RNNPB_Weight::RNNPB_Weight(RNNPB_Neuron* set_from, RNNPB_Neuron* set_to, WeightType set_type)
{
	from=set_from;
	to=set_to;

	constant=false;

	weightType=set_type;

	from->outgoing_weights.push_back(this);
	to->incoming_weights.push_back(this);

	if(to->isRadial==true)
	{
		weight=DEFAULT_INIT_RBFCENTER;
	}
	else
	{
		weight=DEFAULT_INIT_WEIGHT;
	}
	weightUpdate=0.0;

	#ifdef ENABLE_SEPARATE_ANNEALING
	maxWeightUpdate=MAX_WEIGHT_UPDATE;
	minWeightUpdate=MIN_WEIGHT_UPDATE;
	#endif

	learningRate=DEFAULT_WEIGHT_LEARNING_RATE;
	momentum=DEFAULT_WEIGHT_MOMENTUM;

	weightDecay = WEIGHT_DECAY;

	clear();

	network=NULL;

	parsed=0;
}



/*
 * Subklassen-Implementierungen
 * */

/////////////// Normales Gewicht
RNNPB_NormalWeight::RNNPB_NormalWeight(RNNPB_Neuron* set_from, RNNPB_Neuron* set_to) : RNNPB_Weight(set_from, set_to, Normal)
{
	#ifdef ENABLE_DEBUG
	//cout<<"NORMAL WEIGHT CREATED: "<<weightType<<" "<<learningRate<<" "<<momentum<<" "<<weight<<endl;
	#endif
}

/////////////// Gewicht zweiter Ordnung

void RNNPB_SecondOrderWeight::subscribeNetwork(RNNPB_NetworkContainer* set_network, int run)
{
	RNNPB_Weight::subscribeNetwork(set_network, run);

	#ifdef ENABLE_SUBNET_TREE
	if(injection->getNeuronLayer()->parsed==0)
	{		
		//SUBNET: this creates a subnetwork branch
		set_network->registerSubnet(new RNNPB_NetworkContainer(NULL, injection->getNeuronLayer(), NULL)); //TODO: Enable inverse delta buffering for subnets

		injection->getNeuronLayer()->parsed=run; //????
	}
	#else
	if(run==1)	//only register subnets on forward buffering (deltas do not originate from subnets)
		injection->subscribeNetwork(set_network, run);
	#endif
}

double RNNPB_SecondOrderWeight::getWeightedDeltaError(RNNPB_Neuron* caller)
{
	if(caller!=injection)
	{
		return RNNPB_Weight::getWeightedDeltaError(caller);	//rückpropagierung für normales lernen
	}
	else
	{
		//return from->getActivation() * to->getDeltaError();		//rückpropagierung für modulationsupdate

		if(to->isRadial) //RBF-learning
		{
			return learningRate * to->getDeltaError() * (getWeight() - from->getActivation());
		}
		else
		{
			return learningRate * from->getActivation() * to->getDeltaError();
		}		
	}
}

void RNNPB_SecondOrderWeight::learn()
{
	return;	//hier wird nichts gelernt
}

double RNNPB_SecondOrderWeight::getWeight()
{
	return injection->getActivation();
}

RNNPB_SecondOrderWeight::RNNPB_SecondOrderWeight(RNNPB_Neuron* set_from, RNNPB_Neuron* set_to, RNNPB_Neuron* set_injection) : RNNPB_Weight(set_from, set_to, SecondOrder)
{
	clear();

	injection=set_injection;

	injection->outgoing_weights.push_back(this);

	#ifdef ENABLE_DEBUG
	//cout<<"DERIVED: S.O. WEIGHT CREATED: "<<weightType<<" "<<learningRate<<" "<<momentum<<" "<<weight<<endl;
	#endif
}

//PARAMETRIC VERSION:

void RNNPB_SecondOrderWeightCV::learn()
{
	weight = getWeight();	//hypothetisches gewicht für konzept-vektor

	RNNPB_Weight::learn();
}

RNNPB_SecondOrderWeightCV::RNNPB_SecondOrderWeightCV(RNNPB_Neuron* set_from, RNNPB_Neuron* set_to, RNNPB_Neuron* set_injection) : RNNPB_SecondOrderWeight(set_from, set_to, set_injection)
{
	weightType = SecondOrderCV;

	learningRate = DEFAULT_CV_LEARNING_RATE;
	momentum = DEFAULT_CV_MOMENTUM;

	#ifdef ENABLE_SEPARATE_ANNEALING
	maxWeightUpdate=MAX_CV_UPDATE;
	minWeightUpdate=MIN_CV_UPDATE;
	#endif
}




/////////////// Moduliertes Gewicht

double RNNPB_ModulatedWeight::getWeight()
{
	return injection->getActivation() * weight;
}

double RNNPB_ModulatedWeight::getWeightedDeltaError(RNNPB_Neuron* caller)
{
	if(caller!=injection)
	{
		return RNNPB_Weight::getWeightedDeltaError(caller);	//rückpropagierung für normales lernen
	}
	else
	{
		//Anteilige Gewichtsänderung zum Modulator rückpropagieren
		if(to->isRadial) //RBF-learning
		{
			return learningRate * MODULATION_UPDATE_RATE * to->getDeltaError() * (getWeight() - from->getActivation());
		}
		else
		{
			return learningRate * MODULATION_UPDATE_RATE * from->getActivation() * to->getDeltaError();
		}
	}
}

RNNPB_ModulatedWeight::RNNPB_ModulatedWeight(RNNPB_Neuron* set_from, RNNPB_Neuron* set_to, RNNPB_Neuron* set_injection) : RNNPB_SecondOrderWeight(set_from, set_to, set_injection)
{
	weight = 1.0;

	weightType = Modulated;

	//learningRate *= (1.0 - MODULATION_UPDATE_RATE);

	#ifdef ENABLE_DEBUG
	//cout<<"DERIVED: MULTI WEIGHT CREATED: "<<weightType<<" "<<learningRate<<" "<<momentum<<" "<<weight<<endl;
	#endif
}

//PARAMETRIC VERSION:

void RNNPB_ModulatedWeightCV::learn()
{
	double global=weight;
	//double local=injection->getActivation();

	weight=getWeight();
	RNNPB_Weight::learn();	//weightUpdate wird (anhand des lokalen momentums) gelernt	

	double update = weightUpdate * learningRate;

	/*double local_update = MODULATION_UPDATE_RATE * update;
	double global_update = (1.0 - global * MODULATION_UPDATE_RATE) / (MODULATION_UPDATE_RATE + local / update);*/

	pb_weight = weight / (global + update * (1.0 - MODULATION_UPDATE_RATE));
	weight = global + (1.0 - MODULATION_UPDATE_RATE) * update;
}

RNNPB_ModulatedWeightCV::RNNPB_ModulatedWeightCV(RNNPB_Neuron* set_from, RNNPB_Neuron* set_to, RNNPB_Neuron* set_injection) : RNNPB_ModulatedWeight(set_from, set_to, set_injection)
{
	weightType = ModulatedCV;

	learningRate = DEFAULT_CV_LEARNING_RATE;
	momentum = DEFAULT_CV_MOMENTUM;

	weight = 1.0;

	#ifdef ENABLE_SEPARATE_ANNEALING
	maxWeightUpdate=MAX_CV_UPDATE;
	minWeightUpdate=MIN_CV_UPDATE;
	#endif
}





/////////////// Parametric Bias Weight

RNNPB_ParametricBiasWeightCV::RNNPB_ParametricBiasWeightCV(RNNPB_Neuron* set_from, RNNPB_Neuron* set_to): RNNPB_Weight(set_from, set_to, ParametricBiasCV)
{
	momentum = DEFAULT_CV_MOMENTUM;
	learningRate = DEFAULT_CV_LEARNING_RATE;

	weight=1.0;

	#ifdef ENABLE_SEPARATE_ANNEALING
	maxWeightUpdate=MAX_CV_UPDATE;
	minWeightUpdate=MIN_CV_UPDATE;
	#endif

	#ifdef ENABLE_DEBUG
	//cout<<"DERIVED: PARAMETRIC WEIGHT CREATED: "<<weightType<<" "<<learningRate<<" "<<momentum<<" "<<weight<<endl;
	#endif
}




/////////////// Delayed Line

void RNNPB_TapDelayLineWeight::cumulateWeightUpdates()	//Aufsummierung der Gewichtsupdates für Batch-Learning. TODO: Delayed RBFs
{
	#ifdef ENABLE_DEBUG
	if(!deltaErrorTimesActivationSumUptodate)
	{
	#endif
		double tmp=0;

		if(to->isRadial)
		{
			tmp += (getWeight() - outputDelayBuffer) * to->getDeltaError();

			#ifdef ENABLE_BPTT
				tmp += getWeight() * deltaErrorSumBuffer - productSumBuffer;
			#endif

			/* OBSOLETE:
			#ifdef ENABLE_BPTT
			for(i=0;i<outputDelayBufferBuffer.size() && i < BPTT_HORIZONT;i++)		//Berechnung lässt sich durch Ausmultiplizieren vereinfachen!
			{
				tmp += (getWeight() - outputDelayBufferBuffer[i]) * errorDelayBufferBuffer[i];
			}
			#endif*/
		}
		else
		{
			tmp += outputDelayBuffer * to->getDeltaError();

			#ifdef ENABLE_BPTT
				tmp += productSumBuffer;
			#endif

			/* OBSOLETE:
			#ifdef ENABLE_BPTT
			for(i=0;i<fromActivationBuffer.size() && i < BPTT_HORIZONT;i++)		//Berechnung lässt sich vereinfachen: Bei unendlichem horizont einfach summe der produkte erweitern! Kein for nötig!
			{
				tmp += outputDelayBufferBuffer[i] * errorDelayBufferBuffer[i];
			}
			#endif*/
		}

		#ifdef ENABLE_BPTT
			tmp /= iteration+1;
		#endif

		deltaErrorTimesActivationSum += tmp;

		deltaErrorTimesActivationSumUptodate=true;
	#ifdef ENABLE_DEBUG
	}
	else
	{
		cout<<"Error: Mehrfachaufruf?!\n";
	}
	#endif
}

void RNNPB_TapDelayLineWeight::iterate()	//OK
{
	#ifdef ENABLE_BPTT
		/* OBSOLETE
		fromActivationBuffer.push_front(from->getActivation());	//letzten aktuellen wert VORNE anstellen
		toDeltaErrorBuffer.push_front(to->getDeltaError());

		if(fromActivationBuffer.size() > BPTT_HORIZONT)
		{
			fromActivationBuffer.pop_back();						//ältesten wert löschen
			toDeltaErrorBuffer.pop_back();
		}*/

		deltaErrorSumBuffer += to->getDeltaError();
		productSumBuffer += outputDelayBuffer * to->getDeltaError();

		iteration++;
	#endif

	outputDelayBuffer = outputDelayBuffer * (1.0-decay) + from->getActivation() * decay;		//exponentielle mittelung über delaySteps Zeitschritte (siehe konstruktor)
	errorDelayBuffer = errorDelayBuffer * (1.0-decay) + to->getDeltaError() * decay;

	deltaErrorTimesActivationSumUptodate=false;
}

void RNNPB_TapDelayLineWeight::clear()	//OK
{
	//init

	#ifdef ENABLE_BPTT
		//fromActivationBuffer.clear();
		//toDeltaErrorBuffer.clear();
		deltaErrorSumBuffer=0;
		productSumBuffer=0;
		iteration=0;
	#endif

	outputDelayBuffer=0.0;
	errorDelayBuffer=0.0;

	deltaErrorTimesActivationSumUptodate=false;
	deltaErrorTimesActivationSum=0.0;
}

double RNNPB_TapDelayLineWeight::getOutput()
{
	return outputDelayBuffer * getWeight();
}

double RNNPB_TapDelayLineWeight::getWeightedDeltaError(RNNPB_Neuron* caller)
{
	if(to->isRadial)
	{
		return errorDelayBuffer * (outputDelayBuffer - getWeight());
	}
	else
	{
		return errorDelayBuffer * getWeight();
	}
}

RNNPB_TapDelayLineWeight::RNNPB_TapDelayLineWeight(RNNPB_Neuron* set_from, RNNPB_Neuron* set_to, int set_delaySteps) : RNNPB_Weight(set_from, set_to, TapDelayLine)
{
	decay = 1.0/set_delaySteps;

	clear();

	#ifdef ENABLE_SEPARATE_ANNEALING
	//TEST TEST TEST: Deaktiviere untere Schranke!
	maxWeightUpdate=MAX_WEIGHT_UPDATE;
	minWeightUpdate=0;
	#endif

	#ifdef ENABLE_DEBUG
	//cout<<"DERIVED: MULTI WEIGHT CREATED: "<<weightType<<" "<<learningRate<<" "<<momentum<<" "<<weight<<endl;
	#endif
}
