% To run program in Matlab 2020b

clear
clc
close all
warning('off','MATLAB:table:ModifiedAndSavedVarnames');

% select region (Ischgl or Gangelt)
regn = 'Ischgl';
% select number of days by which deaths should be shifted
death_shift = 19;
% If Germany, decide whether or not to use Nowcasting data (1 is yes, 0 is
% no)
nowcast = 1;
reset_death = 1;
seldate     = '2020-05-01';

if (nowcast == 0) && (reset_death == 1)
    error('Makes no sense')
end


data = readtable('Fallzahlen_RKI.csv');% https://www.rki.de/DE/Content/InfAZ/N/Neuartiges_Coronavirus/Daten/Fallzahlen_Kum_Tab.html
data = table2timetable(table(datetime(data.Date_reported,'InputFormat','dd.MM.yyyy'),...
    data.New_cases,data.Cases,data.New_deaths,data.Deaths,'VariableNames',{'Date','New_Cases','Sum_cases','New_deaths','Sum_deaths'}));
data.Properties.DimensionNames{1} = 'Date_reported';

if nowcast == 1
    data2 = readtable('Nowcasting_Zahlen.csv'); % Data taken from here https://www.rki.de/DE/Content/InfAZ/N/Neuartiges_Coronavirus/Projekte_RKI/Nowcasting.html
    data2.Properties.VariableNames{1} = 'Infection_Date';
    data2.Properties.VariableNames{2} = 'New_Cases_nowcast';
    data2 = table2timetable(table(datetime(data2.Infection_Date,'InputFormat','dd.MM.yyyy'),...
        data2.New_Cases_nowcast,'VariableNames',{'Date','New_Cases_nowcast'}));
    data2.Cum_cases_nowcast = cumsum(data2.New_Cases_nowcast);
    data = synchronize(data,data2);
    data.Cumulative_cases = data.Cum_cases_nowcast;
    data = data(timerange(data2.Date(1),data.Date_reported(end)),:);
else 
    data.Cumulative_cases = data.Sum_cases;
end

if reset_death == 0
    T2 = table2timetable(table(data.Date_reported,data.Sum_deaths,...
        'VariableNames',{'Date','Deaths'}));
    TT = lag(T2,-death_shift);
    data = synchronize(data,TT);
    data.Cumulative_deaths = data.Deaths;
else
    T2 = table2timetable(table(data.Date_reported,data.New_deaths,...
        'VariableNames',{'Date','Deaths'})); 
    TT = lag(T2,-death_shift); 
    TT.Cum_deaths = cumsum(TT.Deaths);
    data = synchronize(data,TT);
    data.Cumulative_deaths = data.Cum_deaths;
end
% compute mortatility rate (cond on tested)
mort = data.Cumulative_deaths./data.Cumulative_cases;
data.Var9 = mort;

idx = isnan(data.Var9);
data(idx,:) = [];
% define beginning of plot
date_max = max(data.Date_reported);
TR = timerange('2020-03-01',date_max);
data = data(TR,:);

% Compute standard errors
cfr         = data.Var9(seldate); % read out most recent cfr
n           = data.Cumulative_cases(seldate); % read out most recent number of + cases
cfr_var     = cfr-cfr.^2;
var_corr    = cfr_var*(n/(n-1));
sd          = sqrt(var_corr);
se_cfr      = sd/(sqrt(n));
% same for region
if strcmp(regn,'Ischgl') 
    pos         = 105;
    no          = 627;  
elseif strcmp(regn,'Gangelt')
    pos         = 22;
    no          = 129;       
end
p           = pos/no;
p_var       = p-p.^2;
var_corr2   = p_var*(no/(no-1));
sd2         = sqrt(var_corr2);
se_p        = sd2/(sqrt(no));

ifr         = cfr*p;
se_tot      = sqrt(cfr.^2*se_p.^2+p.^2*se_cfr.^2);
CI          = [ifr-norminv(0.975)*se_tot ifr+norminv(0.975)*se_tot];
disp(['This is the IFR point estimate (in %): ' num2str(ifr*100)]);
disp(['This is the 95% confidence interval (in %): ' num2str(CI*100)]);


% Create time-series plot for IFR
ifr_ser     = timetable(data.Date_reported,zeros(size(data,1),1),zeros(size(data,1),1),...
              zeros(size(data,1),1),zeros(size(data,1),1),'VariableNames',{'ifr','CI_lower','CI_upper','se'});
for i = 1:size(data,1)
    seldate2    = ifr_ser.Time(i);
    cfr2        = data.Var9(seldate2); % read out most recent cfr
    n2          = data.Cumulative_cases(seldate2); % read out most recent number of + cases
    cfr_var2    = (cfr2-cfr2.^2)*(n2/(n2-1));
    se_cfr2     = sqrt(cfr_var2)/(sqrt(n2));
    se_tot2     = sqrt(cfr2.^2*se_p.^2+p.^2*se_cfr2.^2);
    ifr2        = cfr2*p;
    CI2         = [ifr2-norminv(0.975)*se_tot2 ifr2+norminv(0.975)*se_tot2];
    
    ifr_ser.ifr(i)          = ifr2*100;
    ifr_ser.se(i)           = se_tot2;
    ifr_ser.CI_lower(i)     = CI2(1,1)*100;
    ifr_ser.CI_upper(i)     = CI2(1,2)*100;
end

figure
plot([datetime({'2020-04-01'},'InputFormat','yyyy-MM-dd');datetime({'2020-04-01'},'InputFormat','yyyy-MM-dd')],[-1;max(ifr_ser.CI_upper)],'--','Color',[168 168 168]/255)
hold on
plot([datetime({'2020-05-01'},'InputFormat','yyyy-MM-dd');datetime({'2020-05-01'},'InputFormat','yyyy-MM-dd')],[-1;max(ifr_ser.CI_upper)],'--','Color',[168 168 168]/255)
plot([datetime({'2020-06-01'},'InputFormat','yyyy-MM-dd');datetime({'2020-06-01'},'InputFormat','yyyy-MM-dd')],[-1;max(ifr_ser.CI_upper)],'--','Color',[168 168 168]/255)
plot([datetime({'2020-07-01'},'InputFormat','yyyy-MM-dd');datetime({'2020-07-01'},'InputFormat','yyyy-MM-dd')],[-1;max(ifr_ser.CI_upper)],'--','Color',[168 168 168]/255)
plot([datetime({'2020-08-01'},'InputFormat','yyyy-MM-dd');datetime({'2020-08-01'},'InputFormat','yyyy-MM-dd')],[-1;max(ifr_ser.CI_upper)],'--','Color',[168 168 168]/255)
plot([datetime({'2020-09-01'},'InputFormat','yyyy-MM-dd');datetime({'2020-09-01'},'InputFormat','yyyy-MM-dd')],[-1;max(ifr_ser.CI_upper)],'--','Color',[168 168 168]/255)
plot([datetime({'2020-10-01'},'InputFormat','yyyy-MM-dd');datetime({'2020-10-01'},'InputFormat','yyyy-MM-dd')],[-1;max(data.Var9*100)],'--','Color',[168 168 168]/255)
plot(ifr_ser.Time, ifr_ser.CI_lower,'Color',[165 30 55]/255,'Linewidth',1);
plot(ifr_ser.Time, ifr_ser.CI_upper,'Color',[165 30 55]/255,'Linewidth',1);
shade(ifr_ser.Time, ifr_ser.CI_lower,ifr_ser.Time, ifr_ser.CI_upper,...
    'FillType',[1 2; 2 1],'Color',[165 30 55]/255);
plot(ifr_ser.Time,ifr_ser.ifr,'Color',[165 30 55]/255,'Linewidth',2.5)
hold off
hAx=gca;                                        % get axes handle
ixmajor=find(day(data.Date_reported)==1);                        % indices for first of month in time vector
hAx.XTick=data.Date_reported(ixmajor);                        % set ticks appropriately
hAx.XAxis.TickLabelFormat='ddMMM';            % set the format to concise form
hAx.XTickLabel=hAx.XTickLabel;
ixminor=1:size(data,1);                    % ides of each month...
hAx.XAxis.MinorTickValues=data.Date_reported(ixminor);        % set minor tick locations
hAx.XAxis.MinorTick='on';    
if max(ifr_ser.CI_upper)>=3
    yticks(0:1:max(ifr_ser.CI_upper))
else 
    yticks(0:0.25:1.5)
end
DateStrings = {'2020-03-01';'2020-10-01'};
t = datetime(DateStrings,'InputFormat','yyyy-MM-dd');
xlim([t(1) data.Date_reported(end)])
set(gca,'YLim',[-0.05 max(ifr_ser.CI_upper)]);
set(gca,'FontSize',13,'TickLabelInterpreter','latex')
titstr = ['IFR for Germany'];
saveas(gcf,['IFR_Germany_' regn '_deaths' num2str(death_shift)],'epsc')


TR2 = timerange('2020-04-01','2020-06-01');
data = data(TR2,:);

figure
plot(ifr_ser.Time, ifr_ser.CI_lower,'Color',[165 30 55]/255,'Linewidth',1);
hold on
plot(ifr_ser.Time, ifr_ser.CI_upper,'Color',[165 30 55]/255,'Linewidth',1);
shade(ifr_ser.Time, ifr_ser.CI_lower,ifr_ser.Time, ifr_ser.CI_upper,...
    'FillType',[1 2; 2 1],'Color',[165 30 55]/255);
plot(ifr_ser.Time,ifr_ser.ifr,'Color',[165 30 55]/255,'Linewidth',2.5)
hold off
hAx=gca;                                        % get axes handle
ixmajor=find(weekday(data.Date_reported)==6);                        % indices for first of month in time vector
ixmajor=ixmajor([1;3;5;7;9]);
hAx.XTick=data.Date_reported(ixmajor);                        % set ticks appropriately
hAx.XAxis.TickLabelFormat='ddMMM';            % set the format to concise form
hAx.XTickLabel=hAx.XTickLabel;
ixminor=1:size(data,1);                    % ides of each month...
hAx.XAxis.MinorTickValues=data.Date_reported(ixminor);        % set minor tick locations
hAx.XAxis.MinorTick='on';    
if max(ifr_ser.CI_upper)>=3
    yticks(0:1:max(ifr_ser.CI_upper))
else 
    yticks(0:0.25:1.5)
end
DateStrings = {'2020-03-01';'2020-10-01'};
t = datetime(DateStrings,'InputFormat','yyyy-MM-dd');
xlim([datetime('2020-04-01') datetime('2020-06-1')])
set(gca,'YLim',[0 1.5]);
set(gca,'FontSize',13,'TickLabelInterpreter','latex')
titstr = ['IFR for Germany'];
saveas(gcf,['IFR_Germany_' regn '_deaths' num2str(death_shift) '_Zoom'],'epsc')


monthstr = month(datetime(seldate),'shortname');
daynum   = day(datetime(seldate));
seldatestr = [monthstr{1} num2str(daynum)];
dlmwrite(['IFR_' seldatestr '_Germany_' regn '_deaths' num2str(death_shift) '.txt'],[ifr CI])
save(['ifr_ser_Germany_' regn '_deaths' num2str(death_shift) '.mat'],'ifr_ser')

% ifr_ser_short = ifr_ser(TR2,:);
% [m_in,Idx_min]=min(ifr_ser_short.ifr);
% ifr_ser_short(Idx_min,:)
% [m_max,Idx_max]=max(ifr_ser_short.ifr);
% ifr_ser_short(Idx_max,:)