#include <sys/time.h>
#include <ctime>

//#define PIECEWISE_TIME
//#define COMPLETE_TIME
//#define GAP_TIME
//#define VERBOSE

inline double get_wall_time() {
  struct timeval time;
  if (gettimeofday(&time,NULL)){
    return 0;
  }
  return (double)time.tv_sec + (double)time.tv_usec * .000001;
}

inline double get_cpu_time(){
    return (double)clock() / CLOCKS_PER_SEC;
}

enum class SolverStatus {
  kNone = -100,
  kTraining = -1,
  kConverged = 0,
  kConvergedMachinePrecision = 1,
  kNoProgress = 2,
  kMaxNumEpoch = 3,
  kMaxNumIterations = 4,
  kNumericalProblems = 5
};

const char * status_to_string(const SolverStatus status) {
  switch (status) {
    case SolverStatus::kTraining:
      return "Training";
    case SolverStatus::kConverged:
      return "Converged";
    case SolverStatus::kConvergedMachinePrecision:
      return "ConvergedMachinePrecision";
    case SolverStatus::kNoProgress:
      return "NoProgress";
    case SolverStatus::kMaxNumEpoch:
      return "MaxNumEpoch";
    case SolverStatus::kMaxNumIterations:
      return "MaxNumIterations";
    case SolverStatus::kNumericalProblems:
      return "NumericalProblems";
    default:
      return "None";
    }
}

mxArray *mxCreateScalar(double x) {
  mxArray *array = mxCreateDoubleMatrix(1, 1, mxREAL);
  *mxGetPr(array) = x;
  return array;
}

mxArray * createScalarStructArray(void const **fields) {
  void const **iter;
  char const **niter;
  char const **names;
  int numFields = 0;
  mxArray *s;
  mwSize dims [] = {1, 1};

  for (iter = fields; *iter; iter += 2) {
    numFields++;
  }

  names = static_cast<const char **>(
    mxCalloc((std::size_t) numFields, sizeof(char const*)));

  for (iter = fields, niter = names; *iter; iter += 2, niter++) {
    *niter = static_cast<const char *>(*iter);
  }

  s = mxCreateStructArray(2, dims, numFields, names);
  for (iter = fields, niter = names; *iter; iter += 2, niter++) {
    mxSetField(s, 0, *niter, (mxArray*)(*(iter+1))) ;
  }
  return s ;
}

/*template <>
inline void MtlOvaSquaredSolver<float>::ComputeAKi() {
  sgemv(&kTranspose, &dim_n_, &dim_t_, &one_, const_cast<float *>(A_),
    &dim_n_, Ki_, &kIncrement, &zero_, AKi_, &kIncrement);
}*/

/*template <>
inline void MtlOvaSquaredSolver<double>::ComputeAKi() {
  dgemv(&kTranspose, &dim_n_, &dim_t_, &one_, const_cast<double *>(A_),
    &dim_n_, Ki_, &kIncrement, &zero_, AKi_, &kIncrement);
}*/


//inline void saxpyFn(float delta, SdcaSize t) {
//  saxpy(&dim_t_, &delta, AKi_, &kIncrement, Theta_t_, &kIncrement);
//}

/* compute real roots of cubic polynomial with real coefficients
 * (c) 2008, 2009 Nicol N. Schraudolph  - all rights reserved -
 * see Wikipedia article on Cubic_equation
 */
int linear(const double A, const double B, double *x) {
/* returns the number n of real roots of the polynomial
 *     A*x + B = 0
 * side effect: stores the real roots in x[0]..x[n-1]
 * n < 0 if equation is found degenerate (all coefficients zero)
 */
  if (A == 0.0) {
    if (B != 0) return 0;
    else return -1;
  }
  x[0] = -B/A;
  return 1;
}

int quadratic(const double A, const double B, const double C, double *x) {
/* returns the number n of real roots of the polynomial
 *     A*x^2 + B*x + C = 0
 * side effect: stores the real roots in x[0]..x[n-1]
 * n < 0 if equation is found degenerate (all coefficients zero)
 */
  if (A == 0.0) return linear(B, C, x);
  const double det = B*B - 4.0*A*C;
  int n = 0;
  if (det >= 0.0) {
    x[n] = sqrt(det);
    if (det > 0.0) { // two distinct roots
      x[n+1] = -x[n];
      x[n] -= B;
      x[n] /= 2.0*A;
      ++n;
    }
    x[n] -= B;
    x[n] /= 2.0*A;
    ++n;
  }
  return n;
}

int cubic(const double A, const double B, const double C, const double D,
  double *x) {
/* returns the number n of real roots of the polynomial
 *     A*x^3 + B*x^2 + C*x + D=0
 * side effect: stores the real roots in x[0]..x[n-1]
 * n < 0 if equation is found degenerate (all coefficients zero)
 */
  if (A == 0.0) return quadratic(B, C, D, x);
  int n=0;

  const double A3 = (1.0/3.0)/A;
  const double BA3 = B*A3;
  const double q = C*A3 - BA3*BA3;
  const double r = (BA3*C - D)/(2.0*A) - BA3*BA3*BA3;
  const double det = q*q*q + r*r;

  if (det > 0.0) { // single real root
    const double s = sqrt(det);
    x[n] = cbrt(r + s) + cbrt(r - s) - BA3;
    ++n;
  } else if (det == 0.0) { // single + double real root
    const double s = cbrt(r);
    x[n] = 2.0*s - BA3;
    ++n;
    if (s > 0.0) { // not a triple root
      x[n] =-s - BA3;
      ++n;
    }
  } else { // 3 distinct real roots
    const double rho = cbrt(sqrt(r*r - det));
    const double theta = atan2(sqrt(-det), r)/3.0;
    const double spt = rho*cos(theta);
    const double smt = rho*sin(theta)*sqrt(3.0);

    x[n] = 2.0*spt - BA3;
    ++n;
    x[n] = -spt - BA3;
    x[n+1] = x[n];
    x[n] += smt;
    ++n;
    x[n] -= smt;
    ++n;
  }
  return n;
}

template <typename T>
inline T Quartic(const T a, const T b, const T c, const T d, const T x) {
  T x2 = x * x;
  T x3 = x * x2;
  T x4 = x * x3;
  return a * x4 + b * x3 + c * x2 + d * x;
}

template <typename T>
T MinQuartic(const T a, const T b, const T c, const T d) {
  /* Minimize a quartic function
   *  a*x^4 + b*x^3 + c*x^2 + d*x
   * (unconstrained, but assume a > 0,
   *  therefore the global minimum is at a critical point)
   */

  // Find the critical points
  double roots[3];
  int n = cubic(
    4.0 * static_cast<double>(a),
    3.0 * static_cast<double>(b),
    2.0 * static_cast<double>(c),
    static_cast<double>(d),
    roots);

  // Find the min
  if (n == 1) {
    return static_cast<T>(roots[0]);
  } else {
    T x(0), v(0), vmin = std::numeric_limits<T>::infinity();
    for (int i = 0; i < n; ++i) {
      v = Quartic(a, b, c, d, static_cast<T>(roots[i]));
      if (v < vmin) {
        vmin = v;
        x = static_cast<T>(roots[i]);
      }
    }
    return x;
  }
}

template <typename T>
T MinQuartic(const T a, const T b, const T c, const T d,
    const T lb, const T ub) {
  /* Minimize a quartic function
   *  a*x^4 + b*x^3 + c*x^2 + d*x
   * subject to
   *  lb <= x <= ub
   */

  // Find the critical points
  double roots[3];
  int n = cubic(
    4.0 * static_cast<double>(a),
    3.0 * static_cast<double>(b),
    2.0 * static_cast<double>(c),
    static_cast<double>(d),
    roots);

  // Clip the values and return the min
  T x(0), v(0), vmin = std::numeric_limits<T>::infinity();
  double lower = static_cast<double>(lb);
  double upper = static_cast<double>(ub);
  for (int i = 0; i < n; ++i) {
    roots[i] = std::max(lower, std::min(upper, roots[i]));
    v = Quartic(a, b, c, d, static_cast<T>(roots[i]));
    if (v < vmin) {
      vmin = v;
      x = static_cast<T>(roots[i]);
    }
  }
  return x;
}
