/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.helper;

import org.openmuc.jsml.structures.Integer16;
import org.openmuc.jsml.structures.OctetString;
import org.openmuc.jsml.structures.SML_Message;
import org.openmuc.jsml.structures.SML_MessageBody;
import org.openmuc.jsml.structures.SML_PublicCloseReq;
import org.openmuc.jsml.structures.SML_PublicOpenReq;
import org.openmuc.jsml.structures.Unsigned8;

import de.ekut.informatik.kn.josef.myjsml.SML_COSEM_MessageBody;
import de.ekut.informatik.kn.josef.myjsml.SML_ActionCosemReq;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemAttrIndexList;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemAttrList;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemValue;
import de.ekut.informatik.kn.josef.myjsml.SML_GetCosemReq;
import de.ekut.informatik.kn.josef.myjsml.SML_SetCosemReq;

/**
 * Class with static methods to create SML Messages. Provides default constants.
 * 
 * @author Daniel Fuchs
 * @author Michael Hoefling
 */
public class SMLMessageCreator {

    /**
     * Default group no. Only use this for simplicity.
     */
    public static final Unsigned8 GROUPNO_DEFAULT = new Unsigned8(1);
    /**
     * Do not abort on error.
     */
    public static final Unsigned8 ABORT_0 = new Unsigned8(0x00);
    /**
     * Continue with next group on error.
     */
    public static final Unsigned8 ABORT_1 = new Unsigned8(0x01);
    /**
     * Continue current group on error, but no more groups.
     */
    public static final Unsigned8 ABORT_2 = new Unsigned8(0x02);
    /**
     * Abort on error.
     */
    public static final Unsigned8 ABORT_FF = new Unsigned8(0xFF);

    /**
     * SML Version 1 (1.04).
     */
    public static final Unsigned8 VERSION = new Unsigned8(1);

    /**
     * Creates a SML OpenRequest message with authentication.
     * 
     * @param transactionID
     * @param groupNo
     * @param abortOnError
     * @param clientID
     * @param reqFileID
     * @param serverID
     * @param username
     * @param password
     * @return
     */
    public static SML_Message createOpenRequestMessage(
            OctetString transactionID, OctetString clientID,
            OctetString reqFileID, OctetString serverID, OctetString username,
            OctetString password) {
        // OPTIONAL: codepage, serverID, username, password, smlVersion
        SML_PublicOpenReq request = new SML_PublicOpenReq(null, clientID,
                reqFileID, serverID, username, password, VERSION);
        SML_MessageBody body = new SML_MessageBody(SML_MessageBody.OpenRequest,
                request);
        SML_Message message = new SML_Message(transactionID, GROUPNO_DEFAULT,
                ABORT_0, body);
        return message;
    }

    /**
     * Creates a SML CloseRequest message.
     * 
     * @param transactionID
     * @param groupNo
     * @param abortOnError
     * @return
     */
    public static SML_Message createCloseRequestMessage(
            OctetString transactionID) {
        // OPTIONAL: globalSignature
        SML_PublicCloseReq request = new SML_PublicCloseReq(null);
        SML_MessageBody body = new SML_MessageBody(SML_MessageBody.CloseRequest,
                request);
        SML_Message message = new SML_Message(transactionID, GROUPNO_DEFAULT,
                ABORT_0, body);
        return message;
    }

    /**
     * Creates a SML GetCosemRequest message with authentication.
     * 
     * @param transactionID
     * @param groupNo
     * @param abortOnError
     * @param clientID
     * @param serverID
     * @param objName
     * @param classID
     * @param classVersion
     * @param attrIndexList
     * @param username
     * @param password
     * @return
     */
    public static SML_Message createGetCosemRequestMessage(
            OctetString transactionID, OctetString clientID,
            OctetString serverID, OctetString objName, Integer16 classID,
            Integer16 classVersion, SML_CosemAttrIndexList attrIndexList,
            OctetString username, OctetString password) {
        // OPTIONAL: serverID, username, password, attIndexList
        SML_GetCosemReq request = new SML_GetCosemReq(clientID, serverID,
                username, password, objName, classID, classVersion,
                attrIndexList);
        SML_MessageBody body = new SML_COSEM_MessageBody(
                SML_COSEM_MessageBody.GetCosemRequest, request);
        SML_Message message = new SML_Message(transactionID, GROUPNO_DEFAULT,
                ABORT_0, body);
        return message;
    }

    /**
     * Creates a SML SetCosemRequest message with authentication.
     * 
     * @param transactionID
     * @param groupNo
     * @param abortOnError
     * @param clientID
     * @param serverID
     * @param objName
     * @param classID
     * @param classVersion
     * @param attrList
     * @param username
     * @param password
     * @return
     */
    public static SML_Message createSetCosemRequestMessage(
            OctetString transactionID, OctetString clientID,
            OctetString serverID, OctetString objName, Integer16 classID,
            Integer16 classVersion, SML_CosemAttrList attrList,
            OctetString username, OctetString password) {
        // OPTIONAL: serverID, username, password
        SML_SetCosemReq request = new SML_SetCosemReq(clientID, serverID,
                username, password, objName, classID, classVersion, attrList);
        SML_MessageBody body = new SML_COSEM_MessageBody(
                SML_COSEM_MessageBody.SetCosemRequest, request);
        SML_Message message = new SML_Message(transactionID, GROUPNO_DEFAULT,
                ABORT_0, body);
        return message;
    }

    /**
     * Creates a SML ActionCosemRequest message with authentication.
     * 
     * @param transactionID
     * @param groupNo
     * @param abortOnError
     * @param clientID
     * @param serverID
     * @param objName
     * @param classID
     * @param classVersion
     * @param serviceIndex
     * @param serviceParameter
     * @param username
     * @param password
     * @return
     */
    public static SML_Message createActionCosemRequestMessage(
            OctetString transactionID, OctetString clientID,
            OctetString serverID, OctetString objName, Integer16 classID,
            Integer16 classVersion, Unsigned8 serviceIndex,
            SML_CosemValue serviceParameter, OctetString username,
            OctetString password) {
        // OPTIONAL: serverID, username, password, serviceParameter
        SML_ActionCosemReq request = new SML_ActionCosemReq(clientID, serverID,
                username, password, objName, classID, classVersion,
                serviceIndex, serviceParameter);
        SML_MessageBody body = new SML_COSEM_MessageBody(
                SML_COSEM_MessageBody.ActionCosemRequest, request);
        SML_Message message = new SML_Message(transactionID, GROUPNO_DEFAULT,
                ABORT_0, body);
        return message;
    }

}
