/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation
Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.mycosem;

import java.util.ArrayList;

import org.openmuc.jsml.structures.Integer16;
import org.openmuc.jsml.structures.OctetString;

import de.ekut.informatik.kn.josef.myjsml.SML_CosemValue;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemValueList;

/**
 * Representation (simplified) of the COSEM IC ASSOCIATION_LN [classID=15,
 * classVersion=1]. Attributes and methods which are not used in this program
 * are omitted for simplicity. Attribute 1: logicalName. Attribute 2:
 * objectList.
 * 
 * @author Daniel Fuchs
 */
public class AssociationLN extends InterfaceClass {

    /**
     * Object list at index 2.
     */
    protected ObjectListElement[] objectList = new ObjectListElement[] {};

    /**
     * Constructor.
     * 
     * @param logicalName
     */
    public AssociationLN(String logicalName) {
        this.logicalName = new OctetString(logicalName); // evtl nichtmal
                                                         // nötig, wenn fester
                                                         // name
        this.classID = new Integer16((short) 15);
        this.classVersion = new Integer16((short) 1);
    }

    /**
     * Gets the object list.
     * 
     * @return
     */
    public ObjectListElement[] getObjectList() {
        return objectList;
    }

    /**
     * Sets the object list.
     * 
     * @param objectList
     */
    public void setObjectList(ObjectListElement[] objectList) {
        this.objectList = objectList;
    }

    /**
     * Refreshes the object list at index 2, with the objects in the given list.
     * 
     * @param objList
     */
    public void refreshObjectList(ArrayList<InterfaceClass> objList) {
        ObjectListElement[] newObjList = new ObjectListElement[objList.size()];
        for (int i = 0; i < newObjList.length; i++) {
            newObjList[i] = new ObjectListElement(objList.get(i).getClassID(),
                    objList.get(i).getClassVersion(),
                    objList.get(i).getLogicalName());
        }
        this.objectList = newObjList;
    }

    @Override
    public SML_CosemValue getValue(int index) {
        switch (index) {
        case 1:
            return new SML_CosemValue(this.logicalName);
        case 2: {
            SML_CosemValue[] array = new SML_CosemValue[this.objectList.length];
            for (int i = 0; i < array.length; i++) {
                array[i] = this.objectList[i].constructSML();
            }
            SML_CosemValue objList = new SML_CosemValue(
                    new SML_CosemValueList(array));
            return objList;
        }
        default:
            return null;
        }
    }

    @Override
    public boolean setValue(int index, SML_CosemValue value) {
        switch (index) {
        case 1: {
            if (value.getChoice() instanceof OctetString) {
                this.logicalName = (OctetString) value.getChoice();
                return true;
            } else {
                return false;
            }
        }
        case 2: {
            if (!(value.getChoice() instanceof SML_CosemValueList))
                return false;
            SML_CosemValueList lis = (SML_CosemValueList) value.getChoice();
            SML_CosemValue[] array = lis.getArray();
            ObjectListElement[] objList = new ObjectListElement[array.length];
            for (int i = 0; i < array.length; i++) {
                ObjectListElement elem = new ObjectListElement();
                elem.deconstructSML(array[i]);
                objList[i] = elem;
            }
            this.objectList = objList;
            return true;
        }
        default:
            return false;
        }
    }

    @Override
    public boolean runMethod(int index) {
        return false;
    }

    @Override
    public int countAttributes() {
        return 2;
    }

    @Override
    public int countMethods() {
        return 0;
    }

    @Override
    public String printPretty() {
        String s = "";
        s = s + "Object:ASSOCIATION_LN(classID:" + this.classID.getVal()
                + ",classVersion:" + this.classVersion.getVal() + ")";
        s = s + "[logicalName:" + this.logicalName + "]";
        s = s + "[objectList length: " + this.objectList.length;
        for (int i = 0; i < objectList.length; i++) {
            s = s + ", " + objectList[i].printPretty();
        }
        s += "]";
        return s;
    }

    @Override
    public String[] getAttributesAsString() {
        String s2 = "";
        for (int i = 0; i < objectList.length; i++) {
            s2 += "#" + objectList[i].printPretty();
        }
        return new String[] { this.logicalName.toString(), s2 };
    }

}
