/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.mycosem;

import org.openmuc.jsml.structures.Integer32;

import de.ekut.informatik.kn.josef.myjsml.SML_CosemValue;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemValueList;

/**
 * Representation of a scaler unit, as used in COSEM IC (for example Register).
 * 
 * @author Daniel Fuchs
 */
public class ScalerUnit {

    /**
     * Scaler.
     */
    protected Integer32 scaler;

    /**
     * Unit.
     */
    protected Integer32 unit;

    /**
     * Constructor Default if not used: scaler=0, unit=255.
     */
    public ScalerUnit() {
        this.scaler = new Integer32(0);
        this.unit = new Integer32(255);
    }

    /**
     * Constructor
     * 
     * @param scaler
     * @param unit
     */
    public ScalerUnit(Integer32 scaler, Integer32 unit) {
        this.scaler = scaler;
        this.unit = unit;
    }

    public Integer32 getScaler() {
        return scaler;
    }

    public void setScaler(Integer32 scaler) {
        this.scaler = scaler;
    }

    public Integer32 getUnit() {
        return unit;
    }

    public void setUnit(Integer32 unit) {
        this.unit = unit;
    }

    /**
     * Deconstructs a SML_CosemValue, containing a representation a scaler unit.
     * Sets the values in this scaler unit.
     * 
     * @param value
     */
    public boolean deconstructSML(SML_CosemValue value) {
        if (!(value.getChoice() instanceof SML_CosemValueList))
            return false;
        SML_CosemValueList outerList = (SML_CosemValueList) value.getChoice();
        SML_CosemValue[] array = outerList.getArray();
        if (array.length != 2)
            return false;
        boolean success = true;
        if (array[0].getChoice() instanceof Integer32) {
            this.scaler = (Integer32) array[0].getChoice();
        } else {
            success = false;
        }
        if (array[1].getChoice() instanceof Integer32) {
            this.unit = (Integer32) array[1].getChoice();
        } else {
            success = false;
        }
        return success;
    }

    /**
     * Constructs a SML_CosemValue representation of the scaler unit.
     * 
     * @return
     */
    public SML_CosemValue constructSML() {
        SML_CosemValue s = new SML_CosemValue(this.scaler);
        SML_CosemValue u = new SML_CosemValue(this.unit);
        SML_CosemValue[] array = new SML_CosemValue[] { s, u };
        return new SML_CosemValue(new SML_CosemValueList(array));
    }

    /**
     * Returns a nice String with information about the scaler unit.
     * 
     * @return
     */
    public String printPretty() {
        String s = "ScalerUnit[scaler:" + this.scaler.getVal() + ", unit:"
                + this.unit.getVal() + "]";
        return s;
    }
}
