/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.restserver.model;

import de.ekut.informatik.kn.josef.restserver.schema.AttributeType;
import de.ekut.informatik.kn.josef.restserver.schema.AttributesType;
import de.ekut.informatik.kn.josef.restserver.schema.CosemType;
import de.ekut.informatik.kn.josef.restserver.schema.LdevType;
import de.ekut.informatik.kn.josef.restserver.schema.LdevsType;
import de.ekut.informatik.kn.josef.restserver.schema.ObjectType;
import de.ekut.informatik.kn.josef.restserver.schema.ObjectsType;

/**
 * Class that represents a COSEM data model on a RESTful web service server.
 * Contains methods and to get, add and update the data structure.
 * 
 * The COSEM data model is based on, but not exactly identical, to the
 * Specification
 * "Technische Richtlinie BSI TR-03109-1 - Anlage II: COSEM/HTTP Webservices, Version 1.0"
 * 
 * @author Daniel Fuchs
 */
public class RestModel {

    /**
     * Root element, contains COSEM data model.
     */
    private static CosemType cosem = new CosemType();

    /**
     * Contains list of logical devices.
     */
    private static LdevsType ldevs = new LdevsType();

    /**
     * Initializes empty data model.
     */
    public static void init() {
        cosem.setLdevs(ldevs);
        ldevs.getLdev();
    }

    /**
     * Parses the depth of the query from String to integer.
     * 
     * @param depth
     * @return
     */
    public static int parseDepth(String depth) {
        int dep = 0;
        if (depth != null) {
            try {
                dep = Integer.parseInt(depth);
            } catch (NumberFormatException e) {
                // e.printStackTrace();
            }
        } else {
            dep = 1;
        }
        return dep;
    }

    /*
     * +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * +++++++++++++++++++ +++++++++++++++++++++++++++++++++++ Query Getter
     * +++++++++++++++++++++++++++++++++++++++++++
     * +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * +++++++++++++++++++
     */

    /**
     * Gets the logical device list. Depth depends on the query parameter depth.
     * 
     * @param depth
     * @return
     */
    public static LdevsType getLdevsQuery(int depth) {
        LdevsType devs = new LdevsType();
        devs.setCount(getLdevs().getCount());
        if (depth <= 0)
            return devs;
        else {
            depth--;
            for (int i = 0; i < getLdevs().getLdev().size(); i++) {
                devs.getLdev().add(getLdevQuery(depth,
                        getLdevs().getLdev().get(i).getId()));
            }
            return devs;
        }
    }

    /**
     * Gets a logical device, depending on devID. Returns null, if no ldev
     * found. Depth depends on the query parameter depth.
     * 
     * @param depth
     * @param devID
     * @return
     */
    public static LdevType getLdevQuery(int depth, String devID) {
        LdevType dev = new LdevType();
        if (getLdev(devID) == null)
            return null;
        dev.setId(devID);
        dev.setObjects(null);
        if (depth <= 0)
            return dev;
        else {
            depth--;
            dev.setObjects(getObjectsQuery(depth, devID));
            return dev;
        }
    }

    /**
     * Gets an object list, depending on devID. Returns null, if no list found.
     * Depth depends on the query parameter depth.
     * 
     * @param depth
     * @param devID
     * @return
     */
    public static ObjectsType getObjectsQuery(int depth, String devID) {
        ObjectsType objects = new ObjectsType();
        if (getObjects(devID) == null)
            return null;
        objects.setCount(getObjects(devID).getCount());
        if (depth <= 0)
            return objects;
        else {
            depth--;
            for (int i = 0; i < getObjects(devID).getObject().size(); i++) {
                objects.getObject().add(getObjectQuery(depth, devID,
                        getObjects(devID).getObject().get(i).getId()));
            }
            return objects;
        }
    }

    /**
     * Gets an object, depending on the devID and objID. Returns null, if no
     * object found. Depth depends on the query parameter depth.
     * 
     * @param depth
     * @param devID
     * @param objID
     * @return
     */
    public static ObjectType getObjectQuery(int depth, String devID,
            String objID) {
        ObjectType obj = new ObjectType();
        if (getObject(devID, objID) == null)
            return null;
        obj.setId(objID);
        obj.setClassId(getObject(devID, objID).getClassId());
        obj.setVersion(getObject(devID, objID).getVersion());
        obj.setAttributes(null);
        if (depth <= 0)
            return obj;
        else {
            depth--;
            obj.setAttributes(getAttributesQuery(depth, devID, objID));
            return obj;
        }
    }

    /**
     * Gets the attribute list, depending on the devID and objID. Returns null,
     * if no list found. Depth depends on the query parameter depth.
     * 
     * @param depth
     * @param devID
     * @param objID
     * @return
     */
    public static AttributesType getAttributesQuery(int depth, String devID,
            String objID) {
        AttributesType atts = new AttributesType();
        if (getAttributes(devID, objID) == null)
            return null;
        atts.setCount(getAttributes(devID, objID).getCount());
        if (depth <= 0)
            return atts;
        else {
            depth--;
            for (int i = 0; i < getAttributes(devID, objID).getAttribute()
                    .size(); i++) {
                atts.getAttribute()
                        .add(getAttributeQuery(depth, devID, objID,
                                getAttributes(devID, objID).getAttribute()
                                        .get(i).getId() + ""));
            }
            return atts;
        }
    }

    /**
     * Gets an attribute, depending on the devID, objID and attIndex. Returns
     * null, if no attribute found. Depth depends on the query parameter depth.
     * 
     * @param depth
     * @param devID
     * @param objID
     * @param attIndex
     * @return
     */
    public static AttributeType getAttributeQuery(int depth, String devID,
            String objID, String attIndex) {
        AttributeType att = new AttributeType();
        if (getAttribute(devID, objID, attIndex) == null)
            return null;
        int ind = 0;
        try {
            ind = Integer.parseInt(attIndex);
        } catch (Exception e) {
        }
        att.setId(ind);
        att.setValue(null);
        if (depth <= 0)
            return att;
        else {
            depth--;
            att.setValue(getAttribute(devID, objID, attIndex).getValue());
            return att;
        }
    }

    /*
     * +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * +++++++++++++++++++ +++++++++++++++++++++++++++++++++++++++ Getter
     * +++++++++++++++++++++++++++++++++++++++++++++
     * +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * +++++++++++++++++++
     */

    /**
     * Gets the cosem data model.
     * 
     * @return
     */
    public static CosemType getCosem() {
        return cosem;
    }

    /**
     * Gets the logical device list.
     * 
     * @return
     */
    public static LdevsType getLdevs() {
        return ldevs;
    }

    /**
     * Gets a logical device, depending on devID. Returns null, if no ldev
     * found.
     * 
     * @param devID
     * @return
     */
    public static LdevType getLdev(String devID) {
        for (int i = 0; i < getLdevs().getLdev().size(); i++) {
            LdevType ld = getLdevs().getLdev().get(i);
            if (ld.getId().equals(devID)) {
                return ld;
            }
        }
        return null;
    }

    /**
     * Gets an object list, depending on devID. Returns null, if no list found.
     * 
     * @param devID
     * @return
     */
    public static ObjectsType getObjects(String devID) {
        LdevType ldev = getLdev(devID);
        if (ldev != null) {
            return ldev.getObjects();
        } else {
            return null;
        }
    }

    /**
     * Gets an object, depending on the devID and objID. Returns null, if no
     * object found.
     * 
     * @param devID
     * @param objID
     * @return
     */
    public static ObjectType getObject(String devID, String objID) {
        ObjectsType objects = getObjects(devID);
        if (objects != null) {
            for (int i = 0; i < objects.getObject().size(); i++) {
                if ((objects.getObject().get(i).getId() + "").equals(objID)) {
                    return objects.getObject().get(i);
                }
            }
        }
        return null;
    }

    /**
     * Gets the attribute list, depending on the devID and objID. Returns null,
     * if no list found.
     * 
     * @param devID
     * @param objID
     * @return
     */
    public static AttributesType getAttributes(String devID, String objID) {
        ObjectType object = getObject(devID, objID);
        if (object != null) {
            return object.getAttributes();
        } else {
            return null;
        }
    }

    /**
     * Gets an attribute, depending on the devID, objID and attIndex. Returns
     * null, if no attribute found.
     * 
     * @param devID
     * @param objID
     * @param attIndex
     * @return
     */
    public static AttributeType getAttribute(String devID, String objID,
            String attIndex) {
        AttributesType atts = getAttributes(devID, objID);
        if (atts != null) {
            for (int i = 0; i < atts.getAttribute().size(); i++) {
                if ((atts.getAttribute().get(i).getId() + "")
                        .equals(attIndex)) {
                    return atts.getAttribute().get(i);
                }
            }
        }
        return null;
    }

    /*
     * +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * +++++++++++++++++++ +++++++++++++++++++++++++++++++++++++++ Putter
     * +++++++++++++++++++++++++++++++++++++++++++++
     * +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * +++++++++++++++++++
     */

    /**
     * Puts a logical device to the ldev list. If logical device is present,
     * update it.
     * 
     * @param ldev
     */
    public static void putLdev(LdevType ldev) {
        LdevsType ldevs = getLdevs();
        int index = ldevs.contains(ldev);
        if (index < 0) { // ldev is not present -> add it
            ldevs.getLdev().add(ldev);
            ldevs.updateCount();
        } else { // ldev is present -> update its objects
            LdevType ldevOld = ldevs.getLdev().get(index);
            for (int i = 0; i < ldev.getObjects().getObject().size(); i++) {
                // put each object of the new ldev in the old ldev
                putObject(ldev.getObjects().getObject().get(i),
                        ldevOld.getId());
            }
        }
    }

    /**
     * Puts an object into the object list of a specific device (devID). If
     * object is present, update it.
     * 
     * @param object
     * @param devID
     */
    public static void putObject(ObjectType object, String devID) {
        ObjectsType objects = getObjects(devID);
        int index = objects.contains(object);
        if (index < 0) { // object is not present -> add it
            objects.getObject().add(object);
            objects.updateCount();
        } else { // object is present -> update its attributes
            ObjectType objOld = objects.getObject().get(index);
            for (int i = 0; i < object.getAttributes().getAttribute()
                    .size(); i++) {
                // put each attribute of the new object in the old object
                putAttribute(object.getAttributes().getAttribute().get(i),
                        devID, objOld.getId());
            }
        }
    }

    /**
     * Puts an attribute into the attribute list of a specific device (devID)
     * and object (objID). If attribute is present, update it.
     * 
     * @param attribute
     * @param devID
     * @param objID
     */
    public static void putAttribute(AttributeType attribute, String devID,
            String objID) {
        AttributesType atts = getAttributes(devID, objID);
        int index = atts.contains(attribute);
        if (index < 0) { // attribute is not present -> add it
            atts.getAttribute().add(attribute);
            atts.updateCount();
        } else { // attribute is present -> update its value
            AttributeType attOld = atts.getAttribute().get(index);
            if (attribute.getValue() != null) {
                attOld.setValue(attribute.getValue());
            }
        }
    }

}
