/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.smartmeter;

import java.io.File;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.Unmarshaller;

import de.ekut.informatik.kn.josef.smartmeter.config.SimulatorConfigType;
import de.ekut.informatik.kn.josef.smartmeter.ui.CommandLineInterface;

/**
 * Smart Meter Simulator main class.
 * 
 * @author Daniel Fuchs
 * @author Michael Hoefling
 */
public class SmartMeterSimulator {

    /**
     * Main method of the Smart Meter Simulator. Start here. If no argument is
     * given, the default configuration is used.
     * 
     * @param args
     *            location of the XML config file
     */
    public static void main(String[] args) {
        // print some fancy copyright header ...
        System.out.println("jOSEF: Smart Meter Simulator");
        System.out.println("========================================");
        System.out.println(
                "(c) 2015 Daniel Fuchs, Michael Hoefling, Florian Heimgaertner");
        System.out.println("         Chair of Communication Networks");
        System.out.println("         University of Tuebingen");
        System.out.println();

        SimulatorConfigType cfg; // the configuration of the smart meter
                                 // simulator

        if (args.length == 0) { // default config
            cfg = defaultConfig();
            System.out.println("Starting with default configuration.");
        } else { // try to load custom config
            cfg = loadConfigXML(args[0]);
            if (cfg != null) { // && isValid
                if (cfg.isValid()) {
                    // custom config is okay
                    System.out.println("Starting with custom configuration.");
                } else {
                    cfg = defaultConfig();
                    System.out.println(
                            "Invalid config file, parameter invalid. Starting with default configuration.");
                }
            } else {
                cfg = defaultConfig();
                System.out.println(
                        "Invalid config file, location invalid. Starting with default configuration.");
            }
        }

        // start the simulator with the configuration and run the user interface
        CommandLineInterface userInterface = new CommandLineInterface(cfg);
        userInterface.runCLI();
    }

    /**
     * Tries to load a XML configuration file from the specified location.
     * 
     * @param location
     * @return config or null, if not found
     */
    private static SimulatorConfigType loadConfigXML(String location) {
        try {
            // try to load config from XML file
            JAXBContext jc = JAXBContext.newInstance(SimulatorConfigType.class);
            Unmarshaller u = jc.createUnmarshaller();
            Object o = u.unmarshal(new File(location));
            SimulatorConfigType cfg = (SimulatorConfigType) o;
            // System.out.println(cfg.printPretty());
            return cfg;
        } catch (Exception e) {
            System.err.println(
                    "An error occurred during loading the configuration file: "
                            + e.getMessage());
            return null;
        }
    }

    /**
     * Returns the default configuration of the smart meter simulator.
     * 
     * @return config
     */
    private static SimulatorConfigType defaultConfig() {
        SimulatorConfigType cfg = new SimulatorConfigType();
        cfg.setAnnualConsumption(1000);
        cfg.setAnnualInfeed(1000);
        cfg.setUsername(null); // do not set empty string, null is valid here
        cfg.setPassword(null); // do not set empty string, null is valid here
        cfg.setPort(33333);
        cfg.setSecPerQuarterHour(4);
        return cfg;
    }

}
