/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.smartmeter.cosem;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.List;

import org.openmuc.jsml.structures.SML_File;
import org.openmuc.jsml.structures.SML_Message;
import org.openmuc.jsml.structures.SML_MessageBody;
import org.openmuc.jsml.tl.SML_TConnection;
import org.openmuc.jsml.tl.SML_TConnectionListener;

import de.ekut.informatik.kn.josef.myjsml.SML_COSEM_MessageBody;

/**
 * Class that handles incoming connections. SML Files are received, handled and
 * responded.
 * 
 * @author Daniel Fuchs
 */
public class Server implements SML_TConnectionListener {

    /**
     * Handles the requests.
     */
    private RequestHandler handler;

    /**
     * Constructor.
     */
    public Server(RequestHandler handler) {
        super();
        this.handler = handler;
    }

    /**
     * Handles a SML File. Splits it up in its ingredients and handles the SML
     * messages respectively. Besides open and close requests, only COSEM
     * related messages are handled.
     * 
     * @param smlFile
     *            received SML File
     * @param connection
     *            connection over which the SML File was received and the
     *            response will be sent
     */
    public void handleSMLFile(SML_File smlFile, SML_TConnection connection) {
        // stream to build up the SML response file
        ByteArrayOutputStream bs = new ByteArrayOutputStream(); // buf size 100?
        DataOutputStream os = new DataOutputStream(bs);

        List<SML_Message> smlMessages = smlFile.getMessages();
        for (SML_Message message : smlMessages) {
            int tag = message.getMessageBody().getTag().getVal();
            switch (tag) {

            case SML_MessageBody.OpenRequest:
                System.out.println("Received openRequest: handle message.");
                SML_Message openResponseMessage = handler
                        .handleOpenRequest(message);
                try {
                    openResponseMessage.code(os);
                } catch (Exception e) {
                    e.printStackTrace();
                    System.err.println(
                            "There went something wrong with the encoding of a SML openResponse message!");
                }
                break;

            case SML_COSEM_MessageBody.GetCosemRequest:
                System.out.println("Received getCosemRequest: handle message.");
                SML_Message getCosemResponseMessage = handler
                        .handleGetCosemRequest(message);
                // System.out.println("Handled getCosemRequest transID:
                // "+getCosemResponseMessage.getTransactionId().toString());
                try {
                    getCosemResponseMessage.code(os);
                } catch (Exception e) {
                    e.printStackTrace();
                    System.err.println(
                            "There went something wrong with the encoding of a SML getCosemResponse message!");
                }
                break;

            case SML_COSEM_MessageBody.SetCosemRequest:
                System.out.println("Received setCosemRequest: handle message.");
                SML_Message setCosemResponseMessage = handler
                        .handleSetCosemRequest(message);
                try {
                    setCosemResponseMessage.code(os);
                } catch (Exception e) {
                    e.printStackTrace();
                    System.err.println(
                            "There went something wrong with the encoding of a SML setCosemResponse message!");
                }
                break;

            case SML_COSEM_MessageBody.ActionCosemRequest:
                System.out.println(
                        "Received actionCosemRequest: handle message.");
                SML_Message actionCosemResponseMessage = handler
                        .handleActionCosemRequest(message);
                try {
                    actionCosemResponseMessage.code(os);
                } catch (Exception e) {
                    e.printStackTrace();
                    System.err.println(
                            "There went something wrong with the encoding of a SML actionCosemResponse message!");
                }
                break;

            case SML_MessageBody.CloseRequest:
                System.out.println("Received closeRequest: handle message.");
                SML_Message closeResponseMessage = handler
                        .handleCloseRequest(message);
                try {
                    closeResponseMessage.code(os);
                } catch (Exception e) {
                    e.printStackTrace();
                    System.err.println(
                            "There went something wrong with the encoding of a SML closeResponse message!");
                }
                break;

            default:
                System.out.println("Received non-request message: discard.");
            }
        }
        try {
            byte[] binarySML_File = bs.toByteArray(); // binary encoding of the
                                                      // SML File
            connection.send(binarySML_File); // send the file
            System.out.println("SML File successfully sent!");
        } catch (IOException e) {
            e.printStackTrace();
            System.err.println(
                    "There went something wrong with sending the binary SML File over the connection!");
        }
    }

    /**
     * All incoming connections are received here. SML Files are received from
     * the connection and forwarded to the handler.
     */
    @Override
    public void connectionIndication(SML_TConnection connection) {
        try {
            SML_File smlFile = connection.receive();

            System.out.println("Received a SML file from the client...");
            // handle the SML file
            handleSMLFile(smlFile, connection);
        } catch (IOException e) {
            System.err.println("An error occurred during data reception: " + e);
        } finally {
            System.out.println("");
            connection.disconnect();
        }
    }

    /**
     * Indicates if the server stopped listening.
     */
    @Override
    public void serverStoppedListeningIndication(IOException e) {
        System.out.println("Server stopped listening: " + e);
    }

}
