/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.smartmetergateway.cosem;

import java.util.ArrayList;

import org.openmuc.jsml.structures.Integer16;
import org.openmuc.jsml.structures.OctetString;

import de.ekut.informatik.kn.josef.helper.Helper;
import de.ekut.informatik.kn.josef.mycosem.AssociationLN;
import de.ekut.informatik.kn.josef.mycosem.InterfaceClass;
import de.ekut.informatik.kn.josef.mycosem.ObjectListElement;

/**
 * Class that handles filter rules. A client can apply filter rules to define
 * which requests shall be created automatically.
 * 
 * @author Daniel Fuchs
 */
public class Filter {

    /**
     * Filter by the logical name.
     */
    private String[] filterLogicalName = new String[] {};

    /**
     * Filter by the class ID.
     */
    private int[] filterClassID = new int[] {};

    /**
     * Filter by the first byte in an OBIS code, which represents the media.
     * Range [0,15].
     */
    private String[] filterMediaOBIS = new String[] {};

    /**
     * Constructor.
     */
    public Filter() {
        this.filterLogicalName = new String[] {};
        this.filterClassID = new int[] {}; // 1:data , 3:register
        this.filterMediaOBIS = new String[] {}; // 1:electricity, 0:abstract
    }

    /**
     * Clears the filter.
     */
    public void clear() {
        this.filterLogicalName = new String[] {};
        this.filterClassID = new int[] {};
        this.filterMediaOBIS = new String[] {};
    }

    /**
     * Sets the media filter.
     * 
     * @param s
     */
    public void setFilterMedia(String[] s) {
        this.filterMediaOBIS = s;
    }

    /**
     * Sets the class filter.
     * 
     * @param c
     */
    public void setFilterClass(int[] c) {
        this.filterClassID = c;
    }

    /**
     * Builds a list of requests, depending on the filter rules. If no filter
     * rules apply, or the filter is inactive, or no association view is
     * available, then the list is empty.
     * 
     * @param objList
     * @return
     */
    public ArrayList<CosemRequest> applyFilterRules(
            ArrayList<InterfaceClass> objList) {
        ArrayList<CosemRequest> list = new ArrayList<CosemRequest>();
        // find current association
        InterfaceClass c = Helper.findObjectInList(new Integer16((short) 15),
                new OctetString("0.0.40.0.0.255"), objList);
        if (c == null)
            return list; // there is no current association, so there is no
                         // knowledge about objects in the server
        if (!(c instanceof AssociationLN))
            return list; // must be AssociationLN
        AssociationLN association = (AssociationLN) c;
        ObjectListElement[] elems = association.getObjectList();
        if (elems == null || elems.length == 0)
            return list; // if no object list, no knowledge about data model

        // check for filter rules
        for (int i = 0; i < elems.length; i++) {

            // if the object is in one of the filters
            if (filterByClassID(elems[i].getClassID().getVal())
                    || filterByLogicalName(elems[i].getLogicalName().toString())
                    || filterByMediaOBIS(
                            elems[i].getLogicalName().toString())) {
                // create a request for this object
                GetRequest request = new GetRequest(
                        elems[i].getLogicalName().toString(),
                        elems[i].getClassID().getVal(),
                        elems[i].getClassVersion().getVal());
                list.add(request);
            }
        }
        return list;
    }

    /**
     * Checks if the class ID is present in the filter list.
     * 
     * @param classID
     * @return
     */
    private boolean filterByClassID(int classID) {
        for (int i = 0; i < this.filterClassID.length; i++) {
            if (this.filterClassID[i] == classID)
                return true;
        }
        return false;
    }

    /**
     * Checks if the logical name is present in the filter list.
     * 
     * @param logicalName
     * @return
     */
    private boolean filterByLogicalName(String logicalName) {
        for (int i = 0; i < this.filterLogicalName.length; i++) {
            if (this.filterLogicalName[i].equals(logicalName))
                return true;
        }
        return false;
    }

    /**
     * Checks if the first byte in the logical name is present in the filter
     * list.
     * 
     * @param logicalName
     * @return
     */
    private boolean filterByMediaOBIS(String logicalName) {
        if (!logicalName.contains("."))
            return false;
        String s = logicalName.split("\\.")[0];
        for (int i = 0; i < this.filterMediaOBIS.length; i++) {
            if (this.filterMediaOBIS[i].equals(s))
                return true;
        }
        return false;
    }

    /**
     * Returns a nice string of the filter configuration.
     * 
     * @return
     */
    public String printPretty() {
        String s = "Filter[";
        s += "[LogicalName:{";
        for (int i = 0; i < this.filterLogicalName.length; i++) {
            s += "," + this.filterLogicalName[i];
        }
        s += "}][ClassID:{";
        for (int i = 0; i < this.filterClassID.length; i++) {
            s += "," + this.filterClassID[i];
        }
        s += "}][Media:{";
        for (int i = 0; i < this.filterMediaOBIS.length; i++) {
            s += "," + this.filterMediaOBIS[i];
        }
        s += "}]]";
        return s;
    }

}
