/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.smartmetergateway.cosem;

import org.openmuc.jsml.structures.Integer16;
import org.openmuc.jsml.structures.OctetString;
import org.openmuc.jsml.structures.SML_AttentionRes;

import de.ekut.informatik.kn.josef.helper.Helper;
import de.ekut.informatik.kn.josef.mycosem.ServerModel;
import de.ekut.informatik.kn.josef.mycosem.AssociationLN;
import de.ekut.informatik.kn.josef.mycosem.Data;
import de.ekut.informatik.kn.josef.mycosem.InterfaceClass;
import de.ekut.informatik.kn.josef.mycosem.LogicalDevice;
import de.ekut.informatik.kn.josef.mycosem.Register;
import de.ekut.informatik.kn.josef.myjsml.SML_ActionCosemRes;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemAttribute;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemValue;
import de.ekut.informatik.kn.josef.myjsml.SML_GetCosemRes;
import de.ekut.informatik.kn.josef.myjsml.SML_SetCosemRes;

/**
 * Handles SML response messages. Executes services on the COSEM server model on
 * the client side.
 * 
 * @author Daniel Fuchs
 */
public class ResponseHandler {

    /**
     * Server model on the client side.
     */
    private ServerModel serverModel;

    /**
     * Constructor.
     * 
     * @param smc
     */
    public ResponseHandler(ServerModel smc) {
        this.serverModel = smc;
    }

    /**
     * Handles a GET response, coming from a server. Handles each COSEM IC
     * separately. Models the server state based on the retrieved values.
     * 
     * @param response
     */
    public void cosemGetResponseHandler(SML_GetCosemRes response) {
        OctetString serverID = response.getServerId();
        String objName = response.getObjName().toString();
        short classID = response.getClassID().getVal();
        SML_CosemAttribute[] attributes = response.getAttributeList()
                .getCosemAttribute();
        InterfaceClass presentObject = findObjectInLdev(response.getClassID(),
                response.getObjName(), serverID);

        // handle Data
        if (classID == (short) 1) {
            if (presentObject != null) { // the object is already present, just
                                         // update it
                for (int i = 0; i < attributes.length; i++) {
                    if (attributes[i].getAttributeContent()
                            .getChoice() instanceof SML_CosemValue) {
                        SML_CosemValue value = (SML_CosemValue) attributes[i]
                                .getAttributeContent().getChoice();
                        presentObject
                                .setValue(
                                        attributes[i].getAttributeDescription()
                                                .getAttributeIndex().getVal(),
                                        value);
                    }
                }
                System.out.println(presentObject.printPretty());
            } else { // the object is new, put it in list
                Data dataObject = new Data(objName);
                for (int i = 0; i < attributes.length; i++) {
                    if (attributes[i].getAttributeContent()
                            .getChoice() instanceof SML_CosemValue) {
                        SML_CosemValue value = (SML_CosemValue) attributes[i]
                                .getAttributeContent().getChoice();
                        dataObject
                                .setValue(
                                        attributes[i].getAttributeDescription()
                                                .getAttributeIndex().getVal(),
                                        value);
                    }
                }
                serverModel.getLdev(serverID).getObjectList().add(dataObject);
                System.out.println(dataObject.printPretty());

                // check, if data object is logical device name
                if (dataObject.getLogicalName()
                        .equals(new OctetString("0.0.42.0.0.255"))) {
                    if (dataObject.getValue(2)
                            .getChoice() instanceof OctetString) {
                        OctetString name = (OctetString) dataObject.getValue(2)
                                .getChoice();
                        serverModel.getLdev(serverID).setName(name);
                        System.out.println("Logical Device Name [" + name
                                + "] set in LD [" + serverID + "]");
                    }
                }
            }
        }

        // handle Register
        if (classID == (short) 3) {
            if (presentObject != null) { // the object is already present, just
                                         // update it
                for (int i = 0; i < attributes.length; i++) {
                    if (attributes[i].getAttributeContent()
                            .getChoice() instanceof SML_CosemValue) {
                        SML_CosemValue value = (SML_CosemValue) attributes[i]
                                .getAttributeContent().getChoice();
                        presentObject
                                .setValue(
                                        attributes[i].getAttributeDescription()
                                                .getAttributeIndex().getVal(),
                                        value);
                    }
                }
                System.out.println(presentObject.printPretty());
            } else { // the object is new, put it in list
                Register registerObject = new Register(objName);
                for (int i = 0; i < attributes.length; i++) {
                    if (attributes[i].getAttributeContent()
                            .getChoice() instanceof SML_CosemValue) {
                        SML_CosemValue value = (SML_CosemValue) attributes[i]
                                .getAttributeContent().getChoice();
                        registerObject
                                .setValue(
                                        attributes[i].getAttributeDescription()
                                                .getAttributeIndex().getVal(),
                                        value);
                    }
                }
                serverModel.getLdev(serverID).getObjectList()
                        .add(registerObject);
                System.out.println(registerObject.printPretty());
            }
        }

        // handle AssociationLN
        if (classID == (short) 15) {
            if (presentObject != null) { // the object is already present, just
                                         // update it
                for (int i = 0; i < attributes.length; i++) {
                    if (attributes[i].getAttributeContent()
                            .getChoice() instanceof SML_CosemValue) {
                        SML_CosemValue value = (SML_CosemValue) attributes[i]
                                .getAttributeContent().getChoice();
                        presentObject
                                .setValue(
                                        attributes[i].getAttributeDescription()
                                                .getAttributeIndex().getVal(),
                                        value);
                    }
                }
                System.out.println(presentObject.printPretty());
            } else { // the object is new, put it in list
                AssociationLN associationObject = new AssociationLN(objName);
                for (int i = 0; i < attributes.length; i++) {
                    if (attributes[i].getAttributeContent()
                            .getChoice() instanceof SML_CosemValue) {
                        SML_CosemValue value = (SML_CosemValue) attributes[i]
                                .getAttributeContent().getChoice();
                        associationObject
                                .setValue(
                                        attributes[i].getAttributeDescription()
                                                .getAttributeIndex().getVal(),
                                        value);
                    }
                }
                serverModel.getLdev(serverID).getObjectList()
                        .add(associationObject);
                System.out.println(associationObject.printPretty());
            }
        }
    }

    /**
     * Handles a SET response, coming from a server.
     * 
     * @param response
     */
    public void cosemSetResponseHandler(SML_SetCosemRes response) {
        System.out.println("Value set in object ["
                + response.getObjName().toString() + "]");
    }

    /**
     * Handles an ACTION response, coming from a server.
     * 
     * @param response
     */
    public void cosemActionResponseHandler(SML_ActionCosemRes response) {
        System.out.println("Method ran in object ["
                + response.getObjName().toString() + "]");
    }

    /**
     * Handles an ATTENTION response, coming from a server.
     * 
     * @param response
     */
    public void attentionResponseHandler(SML_AttentionRes response) {
        StringBuilder code = new StringBuilder();
        for (byte b : response.getAttentionNo().getOctetString()) {
            code.append(String.format("0x%02x ", b));
        }
        if (response.getAttentionMsg() != null) {
            System.out
                    .println("Attention message [" + response.getAttentionMsg()
                            + "] with code [ " + code + "]");
        } else {
            System.out.println("Attention with code [ " + code + "]");
        }
    }

    /**
     * Finds an object in a logical device.
     * 
     * @param classID
     * @param objName
     * @param serverID
     * @return InterfaceClass if object was found, else null
     */
    private InterfaceClass findObjectInLdev(Integer16 classID,
            OctetString objName, OctetString serverID) {
        LogicalDevice ldev = this.serverModel.getLdev(serverID);
        if (ldev != null) {
            return Helper.findObjectInList(classID, objName,
                    ldev.getObjectList());
        }
        return null;
    }

}
