/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.smartmetergateway.restclient;

import java.util.ArrayList;

import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;

import org.openmuc.jsml.structures.OctetString;

import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.WebResource;

import de.ekut.informatik.kn.josef.mycosem.ServerModel;
import de.ekut.informatik.kn.josef.mycosem.InterfaceClass;
import de.ekut.informatik.kn.josef.restserver.schema.LdevType;

/**
 * Class that represents a REST client that sends data to a RESTful webservice
 * server.
 * 
 * @author Daniel Fuchs
 * @author Michael Hoefling
 */
public class RestClient extends Thread {

    /**
     * Is sending?
     */
    private boolean isSending;

    /**
     * URL of the destination resource on the server.
     */
    private String serverURL;

    private OctetString serverID;

    /**
     * Interval in milliseconds, in which the data is sent. Default is 4000.
     */
    private long interval;

    /**
     * Reference to the client's server model.
     */
    private ServerModel serverModel;

    /**
     * Constructor.
     */
    public RestClient(ServerModel smc) {
        this.isSending = false;
        this.serverURL = "";
        this.interval = 4000;
        this.serverModel = smc;
        this.serverID = smc.getLdevs().get(0).getSap();
        this.start();
    }

    /**
     * Ends the thread.
     */
    public void endThread() {
        this.isSending = false;
        interrupt();
    }

    /**
     * Sets if the REST client is sending.
     * 
     * @param sending
     */
    public void setIsSending(boolean sending) {
        this.isSending = sending;
    }

    /**
     * Sets the server URL.
     * 
     * @param url
     */
    public void setServerURL(String url) {
        this.serverURL = url;
    }

    /**
     * Sets the interval in milliseconds.
     * 
     * @param ms
     */
    public void setInterval(long ms) {
        this.interval = ms;
    }

    /**
     * Method that actually sends data to the REST server. All objects in the
     * list are converted and sent.
     * 
     * @param objList
     *            object list of the source
     * @param ldevID
     *            logical device name of the source
     */
    public void send(ArrayList<InterfaceClass> objList, String ldevID) {
        if (ldevID.equals("EmptyName")) {
            System.out.println(
                    "RestClient sending canceled, logical device name required!");
            return;
        }
        try {
            LdevType ldev = RestClientHelper.createLdev(ldevID, objList);
            Client client = Client.create(); // config
            WebResource service = client
                    .resource(UriBuilder.fromUri(serverURL).build());
            ClientResponse response = service.type(MediaType.APPLICATION_XML)
                    .put(ClientResponse.class, ldev);
            System.out.println("RestClient received response code: "
                    + response.getStatus());
        } catch (Exception e) {
            // e.printStackTrace();
            System.out.println("RestClient connection to URL refused!");
        }
    }

    /**
     * Runs the REST client's thread.
     */
    @Override
    public void run() {
        while (!isInterrupted()) {
            if (isSending) {
                // default: send all electricity objects, but nothing more
                send(serverModel.getElectricityObjects(serverID),
                        serverModel.getLdev(serverID).getName().toString());
                // optional: send also time if desired
            }
            try {
                Thread.sleep(interval);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    };

}
