/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.smartmetergateway.restclient;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.openmuc.jsml.structures.ASNObject;
import org.openmuc.jsml.structures.Integer16;
import org.openmuc.jsml.structures.Integer32;
import org.openmuc.jsml.structures.Integer64;
import org.openmuc.jsml.structures.Integer8;
import org.openmuc.jsml.structures.OctetString;
import org.openmuc.jsml.structures.SML_Boolean;
import org.openmuc.jsml.structures.Unsigned16;
import org.openmuc.jsml.structures.Unsigned32;
import org.openmuc.jsml.structures.Unsigned64;
import org.openmuc.jsml.structures.Unsigned8;

import de.ekut.informatik.kn.josef.mycosem.InterfaceClass;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemValue;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemValueList;
import de.ekut.informatik.kn.josef.restserver.schema.*;

/**
 * Helper class that converts between data models ("SML-COSEM" to "REST-COSEM").
 * Required for the REST-client.
 * 
 * @author Daniel Fuchs
 */
public class RestClientHelper {

    /**
     * Creates a ldev from a logical device ID and a list of ICs.
     * 
     * @param id
     * @param objList
     * @return
     */
    public static LdevType createLdev(String id,
            ArrayList<InterfaceClass> objList) {
        LdevType ldev = new LdevType();
        ldev.setId(id);
        ldev.setObjects(createObjects(objList));
        return ldev;
    }

    /**
     * Creates an object list from a list of ICs.
     * 
     * @param objList
     * @return
     */
    public static ObjectsType createObjects(ArrayList<InterfaceClass> objList) {
        ObjectsType objects = new ObjectsType();

        Iterator<InterfaceClass> it = objList.iterator();
        InterfaceClass c;
        while (it.hasNext()) {
            c = it.next();
            ObjectType object = createObject(c);
            objects.getObject().add(object);
        }

        objects.updateCount();
        return objects;
    }

    /**
     * Creates an object from an IC.
     * 
     * @param obj
     * @return
     */
    public static ObjectType createObject(InterfaceClass obj) {
        switch (obj.getClassID().getVal()) {
        case 1: // DATA object
            de.ekut.informatik.kn.josef.mycosem.Data data = (de.ekut.informatik.kn.josef.mycosem.Data) obj;
            return createObjectData(data);
        case 3: // REGISTER object
            de.ekut.informatik.kn.josef.mycosem.Register register = (de.ekut.informatik.kn.josef.mycosem.Register) obj;
            return createObjectRegister(register);
        default:
            return new ObjectType();
        }
    }

    /**
     * Creates an object from an IC Data.
     * 
     * @param data
     * @return
     */
    public static ObjectType createObjectData(
            de.ekut.informatik.kn.josef.mycosem.Data data) {
        ObjectType object = new ObjectType();
        object.setClassId(data.getClassID().getVal());
        object.setVersion(data.getClassVersion().getVal());
        object.setId(data.getLogicalName().toString());
        object.setAttributes(createAttributes(data));
        return object;
    }

    /**
     * Creates an object from an IC Register.
     * 
     * @param register
     * @return
     */
    public static ObjectType createObjectRegister(
            de.ekut.informatik.kn.josef.mycosem.Register register) {
        ObjectType object = new ObjectType();
        object.setClassId(register.getClassID().getVal());
        object.setVersion(register.getClassVersion().getVal());
        object.setId(register.getLogicalName().toString());
        object.setAttributes(createAttributes(register));
        return object;
    }

    /**
     * Creates an attribute list from an IC Data.
     * 
     * @param data
     * @return
     */
    public static AttributesType createAttributes(
            de.ekut.informatik.kn.josef.mycosem.Data data) {
        AttributesType atts = new AttributesType();
        AttributeType att1 = createAttribute(1,
                data.getLogicalName().toString());
        atts.getAttribute().add(att1);
        AttributeType att2 = createAttribute(2, data.getValue(2));
        atts.getAttribute().add(att2);
        atts.updateCount();
        return atts;
    }

    /**
     * Creates an attribute list from an IC Register.
     * 
     * @param register
     * @return
     */
    public static AttributesType createAttributes(
            de.ekut.informatik.kn.josef.mycosem.Register register) {
        AttributesType atts = new AttributesType();
        AttributeType att1 = createAttribute(1,
                register.getLogicalName().toString());
        atts.getAttribute().add(att1);
        AttributeType att2 = createAttribute(2, register.getValue(2));
        atts.getAttribute().add(att2);
        AttributeType att3 = createAttribute(3, register.getValue(3));
        atts.getAttribute().add(att3);
        atts.updateCount();
        return atts;
    }

    /**
     * Creates an attribute.
     * 
     * @param index
     * @param s
     * @return
     */
    public static AttributeType createAttribute(int index, String s) {
        AttributeType att = new AttributeType();
        att.setId(index);
        CosemValue value = new CosemValue();
        value.setString(s);
        att.setValue(value);
        return att;
    }

    /**
     * Creates an attribute.
     * 
     * @param index
     * @param val
     * @return
     */
    public static AttributeType createAttribute(int index, SML_CosemValue val) {
        AttributeType att = new AttributeType();
        att.setId(index);
        CosemValue value = new CosemValue();
        value = RestClientHelper.convert(val);
        att.setValue(value);
        return att;
    }

    /**
     * Converts a SML_CosemValue to a CosemValue. SML_CosemValue is used in the
     * SML related libraries. CosemValue is used in the RESTful related
     * libraries.
     * 
     * @param value
     * @return
     */
    public static CosemValue convert(SML_CosemValue value) {
        CosemValue result = new CosemValue();
        ASNObject val = value.getChoice();

        if (val instanceof SML_Boolean) {
            SML_Boolean tmp = (SML_Boolean) val;
            boolean v = tmp.getVal();
            result.setBool(v);
            return result;
        }
        if (val instanceof OctetString) {
            OctetString tmp = (OctetString) val;
            String v = tmp.toString();
            result.setString(v);
            return result;
        }

        if (val instanceof Integer8) {
            Integer8 tmp = (Integer8) val;
            byte v = tmp.getVal();
            result.setByte(v);
            return result;
        }
        if (val instanceof Integer16) {
            Integer16 tmp = (Integer16) val;
            short v = tmp.getVal();
            result.setShort(v);
            return result;
        }
        if (val instanceof Integer32) {
            Integer32 tmp = (Integer32) val;
            int v = tmp.getVal();
            result.setInt(v);
            return result;
        }
        if (val instanceof Integer64) {
            Integer64 tmp = (Integer64) val;
            long v = tmp.getVal();
            result.setLong(v);
            return result;
        }

        if (val instanceof Unsigned8) {
            Unsigned8 tmp = (Unsigned8) val;
            int v = tmp.getVal();
            result.setUbyte((byte) v);
            return result;
        }
        if (val instanceof Unsigned16) {
            Unsigned16 tmp = (Unsigned16) val;
            int v = tmp.getVal();
            result.setUshort((short) v);
            return result;
        }
        if (val instanceof Unsigned32) {
            Unsigned32 tmp = (Unsigned32) val;
            int v = tmp.getVal();
            result.setUint(v);
            return result;
        }
        if (val instanceof Unsigned64) {
            Unsigned64 tmp = (Unsigned64) val;
            long v = tmp.getVal();
            result.setUlong(v);
            return result;
        }

        if (val instanceof SML_CosemValueList) {
            CosemValueList cosemValueList = new CosemValueList();
            List<CosemValue> valueList = cosemValueList.getValue();
            SML_CosemValueList tmp = (SML_CosemValueList) val;
            SML_CosemValue[] tmpList = tmp.getArray();
            for (int i = 0; i < tmpList.length; i++) {
                CosemValue tmpValue = RestClientHelper.convert(tmpList[i]);
                if (tmpValue != null) {
                    valueList.add(tmpValue);
                }
            }
            result.setValuelist(cosemValueList);
            return result;
        }

        return result;
    }

}
