/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.smartmetergateway.ui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.net.InetAddress;
import java.net.UnknownHostException;

import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.JScrollPane;
import javax.swing.JSpinner;
import javax.swing.JTabbedPane;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.ScrollPaneConstants;
import javax.swing.SpinnerModel;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingConstants;
import javax.swing.Timer;
import javax.swing.UIManager;
import javax.swing.border.TitledBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import org.openmuc.jsml.structures.Unsigned64;

import de.ekut.informatik.kn.josef.mycosem.InterfaceClass;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemValue;
import de.ekut.informatik.kn.josef.smartmetergateway.cosem.ActionRequest;
import de.ekut.informatik.kn.josef.smartmetergateway.cosem.Client;
import de.ekut.informatik.kn.josef.smartmetergateway.cosem.GetRequest;
import de.ekut.informatik.kn.josef.smartmetergateway.cosem.SetRequest;

/**
 * GUI of a client-server connection within the gateway. The GUI was built with
 * the Eclipse-Plugin WindowBuilder.
 * 
 * @author Daniel Fuchs
 * @author Michael Hoefling
 */
public class ClientTab extends JPanel {

    /**
     * Serial.
     */
    private static final long serialVersionUID = 1L;

    /**
     * Timer to refresh components. Start only, when client is initialized.
     */
    private Timer timer = new Timer(250, new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
            if (client != null && !client.getServerObjectList().isEmpty()) {
                objectListModel.updateList(client.getServerObjectList());
                attributeListModel.updateList(objectJList.getSelectedValue());
            }
        }
    });

    /**
     * The client which is controlled by this GUI.
     */
    private Client client;

    /**
     * List model of COSEM objects.
     */
    private ObjectListModel objectListModel = new ObjectListModel();

    /**
     * List model of COSEM attributes.
     */
    private AttributeListModel attributeListModel = new AttributeListModel();

    /**
     * List of objects. Represented as COSEM objects.
     */
    private JList<InterfaceClass> objectJList = new JList<InterfaceClass>(
            objectListModel);

    /**
     * List of attributes. Represented as Strings.
     */
    private JList<String> attributeJList = new JList<String>(
            attributeListModel);

    /**
     * Chart panel that allows to draw a diagram.
     */
    private CosemObjectChartPanel chartPanel;

    // Servers settings fields
    private JTextField textFieldRestUrl;
    private JTextField textFieldServerIp;
    private JTextField textFieldServerPort;
    private JTextField textFieldServerId;
    private JTextField textFieldClientId;
    private JTextField textFieldUsername;
    private JPasswordField passwordField;

    // Text fields
    private JTextField textFieldManReqParam = new JTextField();

    // Buttons
    final JButton btnConnectServer = new JButton("Connect");
    final JButton btnDisconnectServer = new JButton("Disconnect");
    final JButton btnConnectRest = new JButton("Start Forwarding");
    final JButton btnDisconnectRest = new JButton("Stop Forwarding");
    final JSpinner spinnerIntervalServer = new JSpinner();

    // Filter settings
    JComboBox<String> comboBoxFilterMedia = new JComboBox<String>();
    JComboBox<String> comboBoxFilterClass = new JComboBox<String>();
    final JCheckBox chckbxFilterMedia = new JCheckBox("Filter by medium?");
    final JCheckBox chckbxFilterClass = new JCheckBox("Filter by class id?");

    /**
     * Create the client's panel.
     */
    public ClientTab() {
        setLayout(new GridLayout(1, 1, 0, 0));
        JTabbedPane tabbedPane = new JTabbedPane(JTabbedPane.TOP);
        add(tabbedPane);
        JPanel clientPanel = new JPanel();
        tabbedPane.addTab("Client View", null, clientPanel, null);
        clientPanel.setLayout(new GridLayout(0, 3, 5, 5));

        GridBagLayout gridBagLayout = new GridBagLayout();
        gridBagLayout.rowHeights = new int[] { 0, 0 };
        gridBagLayout.columnWidths = new int[] { 0 };
        gridBagLayout.columnWeights = new double[] { 1.0, 1.0 };
        gridBagLayout.rowWeights = new double[] { 1.0, 1.0 };
        clientPanel.setLayout(gridBagLayout);

        JPanel leftPanel = new JPanel();
        GridBagConstraints gbc_leftPanel = new GridBagConstraints();
        gbc_leftPanel.insets = new Insets(0, 0, 5, 5);
        gbc_leftPanel.fill = GridBagConstraints.BOTH;
        gbc_leftPanel.gridx = 0;
        gbc_leftPanel.gridy = 0;
        gbc_leftPanel.weightx = 0.66;
        gbc_leftPanel.weighty = 0.66;
        clientPanel.add(leftPanel, gbc_leftPanel);
        leftPanel.setLayout(new GridLayout(2, 1, 0, 0));

        JPanel rightPanel = new JPanel();
        GridBagConstraints gbc_rightPanel = new GridBagConstraints();
        gbc_rightPanel.anchor = GridBagConstraints.EAST;
        gbc_rightPanel.insets = new Insets(0, 0, 5, 0);
        gbc_rightPanel.fill = GridBagConstraints.BOTH;
        gbc_rightPanel.gridx = 1;
        gbc_rightPanel.gridy = 0;
        gbc_rightPanel.weightx = 0.34;
        gbc_rightPanel.weighty = 0.66;
        clientPanel.add(rightPanel, gbc_rightPanel);

        // object list panel
        JScrollPane scrollPaneObjects = new JScrollPane();
        scrollPaneObjects
                .setToolTipText("List of all visible objects to this client");
        scrollPaneObjects
                .setViewportBorder(new TitledBorder(null, "Object List",
                        TitledBorder.LEADING, TitledBorder.TOP, null, null));
        leftPanel.add(scrollPaneObjects);
        scrollPaneObjects.setVerticalScrollBarPolicy(
                ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
        scrollPaneObjects.setHorizontalScrollBarPolicy(
                ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS);
        objectJList.setToolTipText(
                "Shows the retrieved objects from the smart meter simulator");

        objectJList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        objectJList.addListSelectionListener(new ListSelectionListener() {
            @Override
            public void valueChanged(ListSelectionEvent e) {
                if (e.getValueIsAdjusting())
                    return;
                attributeListModel.updateList(objectJList.getSelectedValue());
            }
        });
        scrollPaneObjects.setViewportView(objectJList);

        // attribute list panel
        JScrollPane scrollPaneAttributes = new JScrollPane();
        scrollPaneAttributes
                .setViewportBorder(new TitledBorder(null, "Attribute List",
                        TitledBorder.LEADING, TitledBorder.TOP, null, null));
        scrollPaneAttributes.setToolTipText(
                "List of all visible attributes to this client");
        scrollPaneAttributes.setVerticalScrollBarPolicy(
                ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
        scrollPaneAttributes.setHorizontalScrollBarPolicy(
                ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS);
        leftPanel.add(scrollPaneAttributes);
        attributeJList.setToolTipText(
                "Shows the retrieved attributes inside the objects");
        attributeJList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        scrollPaneAttributes.setViewportView(attributeJList);
        rightPanel.setLayout(new BorderLayout(0, 0));

        // server settings panel
        JPanel serverPanel = new JPanel();
        serverPanel.setBorder(
                new TitledBorder(UIManager.getBorder("TitledBorder.border"),
                        "Smart Meter Simulator Settings", TitledBorder.LEFT,
                        TitledBorder.TOP, null, new Color(0, 0, 0)));
        rightPanel.add(serverPanel, BorderLayout.CENTER);
        serverPanel.setLayout(new GridLayout(7, 1, 0, 5));

        JPanel serverIpPanel = new JPanel();
        serverPanel.add(serverIpPanel);
        serverIpPanel.setLayout(new GridLayout(0, 2, 5, 0));

        JLabel lblServerIp = new JLabel("Server IP");
        lblServerIp.setHorizontalAlignment(SwingConstants.RIGHT);
        serverIpPanel.add(lblServerIp);

        textFieldServerIp = new JTextField();
        textFieldServerIp.setText("localhost");
        serverIpPanel.add(textFieldServerIp);
        textFieldServerIp.setColumns(15);

        JPanel serverPortPanel = new JPanel();
        serverPanel.add(serverPortPanel);
        serverPortPanel.setLayout(new GridLayout(0, 2, 5, 0));

        JLabel lblServerPort = new JLabel("Server Port");
        lblServerPort.setHorizontalAlignment(SwingConstants.RIGHT);
        serverPortPanel.add(lblServerPort);

        textFieldServerPort = new JTextField();
        textFieldServerPort.setColumns(15);
        textFieldServerPort.setText("33333");
        serverPortPanel.add(textFieldServerPort);

        JPanel serverIdPanel = new JPanel();
        serverPanel.add(serverIdPanel);
        serverIdPanel.setLayout(new GridLayout(0, 2, 5, 0));

        JLabel lblServerId = new JLabel("Server ID");
        lblServerId.setHorizontalAlignment(SwingConstants.RIGHT);
        serverIdPanel.add(lblServerId);

        textFieldServerId = new JTextField();
        textFieldServerId.setText("1");
        serverIdPanel.add(textFieldServerId);
        textFieldServerId.setColumns(15);

        JPanel clientIdPanel = new JPanel();
        serverPanel.add(clientIdPanel);
        clientIdPanel.setLayout(new GridLayout(0, 2, 5, 0));

        JLabel lblClientId = new JLabel("Client ID");
        lblClientId.setHorizontalAlignment(SwingConstants.RIGHT);
        clientIdPanel.add(lblClientId);

        textFieldClientId = new JTextField();
        textFieldClientId.setText("10");
        clientIdPanel.add(textFieldClientId);
        textFieldClientId.setColumns(15);

        JPanel usernamePanel = new JPanel();
        serverPanel.add(usernamePanel);
        usernamePanel.setLayout(new GridLayout(0, 2, 5, 0));

        JLabel lblUsername = new JLabel("Username");
        lblUsername.setHorizontalAlignment(SwingConstants.RIGHT);
        usernamePanel.add(lblUsername);

        textFieldUsername = new JTextField();
        usernamePanel.add(textFieldUsername);
        textFieldUsername.setColumns(15);

        JPanel passwordPanel = new JPanel();
        serverPanel.add(passwordPanel);
        passwordPanel.setLayout(new GridLayout(0, 2, 5, 0));

        JLabel lblPassword = new JLabel("Password");
        lblPassword.setHorizontalAlignment(SwingConstants.RIGHT);
        passwordPanel.add(lblPassword);

        passwordField = new JPasswordField();
        passwordField.setColumns(15);
        passwordPanel.add(passwordField);

        JPanel serverButtonPanel = new JPanel();
        serverPanel.add(serverButtonPanel);
        serverButtonPanel.setLayout(new GridLayout(1, 2, 0, 0));

        btnConnectServer.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                if (initClient()) {
                    setServerSettingsEditable(false);
                    btnConnectServer.setEnabled(false);
                    btnDisconnectServer.setEnabled(true);
                    btnConnectRest.setEnabled(true);
                } else {
                    // init not successful
                    System.out.println(
                            "Initialization was not successful, invalid parameter(s).");
                }
            }
        });
        serverButtonPanel.add(btnConnectServer);

        btnDisconnectServer.setEnabled(false);
        btnDisconnectServer.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                disconnect();
                setServerSettingsEditable(true);
                btnConnectServer.setEnabled(true);
                btnDisconnectServer.setEnabled(false);
                btnConnectRest.setEnabled(false);
                btnDisconnectRest.setEnabled(false);
            }
        });
        serverButtonPanel.add(btnDisconnectServer);

        // paint panel
        JPanel paintPanel = new JPanel();
        rightPanel.add(paintPanel, BorderLayout.NORTH);
        paintPanel.setBorder(
                new TitledBorder(UIManager.getBorder("TitledBorder.border"),
                        "Visualization", TitledBorder.LEADING, TitledBorder.TOP,
                        null, new Color(0, 0, 0)));

        JButton btnPaint = new JButton("Start painting diagram");
        btnPaint.setToolTipText(
                "Draws a diagram based on the selected object in the list");
        btnPaint.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent arg0) {
                drawDiagram();
            }
        });
        paintPanel.setLayout(new GridLayout(1, 1, 0, 0));
        paintPanel.add(btnPaint);

        // chart panel
        chartPanel = new CosemObjectChartPanel("Chart Panel");
        chartPanel.setBorder(new TitledBorder(
                UIManager.getBorder("TitledBorder.border"),
                "Visualization of selected object", TitledBorder.LEADING,
                TitledBorder.TOP, null, new Color(0, 0, 0)));
        tabbedPane.addTab("Diagram", null, chartPanel, null);

        // request panel
        JPanel requestPanel = new JPanel();
        GridBagConstraints gbc_requestPanel = new GridBagConstraints();
        gbc_requestPanel.anchor = GridBagConstraints.SOUTH;
        gbc_requestPanel.insets = new Insets(0, 0, 0, 5);
        gbc_requestPanel.fill = GridBagConstraints.BOTH;
        gbc_requestPanel.gridx = 0;
        gbc_requestPanel.gridy = 1;
        clientPanel.add(requestPanel, gbc_requestPanel);
        requestPanel.setLayout(new GridLayout(1, 2, 0, 0));

        JPanel manualRequestPanel = new JPanel();
        requestPanel.add(manualRequestPanel);
        manualRequestPanel.setBorder(
                new TitledBorder(UIManager.getBorder("TitledBorder.border"),
                        "Manual Request", TitledBorder.LEADING,
                        TitledBorder.TOP, null, new Color(0, 0, 0)));
        manualRequestPanel.setLayout(new GridLayout(4, 1, 0, 5));

        JPanel manReqChoosePanel = new JPanel();
        manualRequestPanel.add(manReqChoosePanel);
        manReqChoosePanel.setLayout(new GridLayout(1, 2, 5, 0));

        JLabel lblChooseRequestType = new JLabel("Choose request type");
        lblChooseRequestType.setHorizontalAlignment(SwingConstants.RIGHT);
        manReqChoosePanel.add(lblChooseRequestType);

        final JComboBox<String> comboBoxRequests = new JComboBox<String>();
        comboBoxRequests.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent arg0) {
                if (comboBoxRequests.getSelectedItem().equals("GET")) {
                    textFieldManReqParam.setEnabled(false);
                }
                if (comboBoxRequests.getSelectedItem().equals("SET")) {
                    textFieldManReqParam.setEnabled(true);
                }
                if (comboBoxRequests.getSelectedItem().equals("ACTION")) {
                    textFieldManReqParam.setEnabled(false);
                }
            }
        });
        comboBoxRequests.setModel(new DefaultComboBoxModel<String>(
                new String[] { "GET", "SET", "ACTION" }));
        comboBoxRequests.setSelectedIndex(0);
        manReqChoosePanel.add(comboBoxRequests);

        JPanel manReqIndexPanel = new JPanel();
        manualRequestPanel.add(manReqIndexPanel);
        manReqIndexPanel.setLayout(new GridLayout(1, 2, 5, 0));

        JLabel lblSelectIndex = new JLabel("Select index");
        lblSelectIndex.setHorizontalAlignment(SwingConstants.RIGHT);
        manReqIndexPanel.add(lblSelectIndex);

        final JSpinner spinnerSelectIndex = new JSpinner();
        // spinnerSelectIndex.setModel(new SpinnerNumberModel(new Short((short)
        // 1), new Short((short) 1), new Short((short) 3), new Short((short)
        // 1)));
        spinnerSelectIndex.setModel(new SpinnerNumberModel(
                Short.valueOf((short) 1), Short.valueOf((short) 1),
                Short.valueOf((short) 3), Short.valueOf((short) 1)));
        manReqIndexPanel.add(spinnerSelectIndex);

        JPanel manReqParamPanel = new JPanel();
        manualRequestPanel.add(manReqParamPanel);
        manReqParamPanel.setLayout(new GridLayout(1, 2, 5, 0));

        JLabel lblEnterParameter = new JLabel("Enter parameter");
        lblEnterParameter.setHorizontalAlignment(SwingConstants.RIGHT);
        manReqParamPanel.add(lblEnterParameter);

        manReqParamPanel.add(textFieldManReqParam);
        textFieldManReqParam.setColumns(10);

        JPanel manReqSendPanel = new JPanel();
        manualRequestPanel.add(manReqSendPanel);
        manReqSendPanel.setLayout(new GridLayout(1, 2, 5, 0));

        JButton btnSendManReq = new JButton("Send request");
        btnSendManReq.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                SpinnerModel numberModel = spinnerSelectIndex.getModel();
                if (comboBoxRequests.getSelectedItem().equals("GET")) {
                    addManualGetRequest(((SpinnerNumberModel) numberModel)
                            .getNumber().shortValue());
                }
                if (comboBoxRequests.getSelectedItem().equals("SET")) {
                    addManualSetRequest(
                            ((SpinnerNumberModel) numberModel).getNumber()
                                    .shortValue(),
                            textFieldManReqParam.getText());
                }
                if (comboBoxRequests.getSelectedItem().equals("ACTION")) {
                    addManualActionRequest(((SpinnerNumberModel) numberModel)
                            .getNumber().shortValue());
                }
            }
        });
        manReqSendPanel.add(btnSendManReq);

        JPanel automaticRequestPanel = new JPanel();
        requestPanel.add(automaticRequestPanel);
        automaticRequestPanel
                .setBorder(new TitledBorder(null, "Automatic Requests",
                        TitledBorder.LEADING, TitledBorder.TOP, null, null));
        automaticRequestPanel.setLayout(new GridLayout(4, 1, 0, 5));

        final JCheckBox chckbxAutomaticallyRetrieve = new JCheckBox(
                "Retrieve data from Smart Meter Simulator?");
        automaticRequestPanel.add(chckbxAutomaticallyRetrieve);
        chckbxAutomaticallyRetrieve.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent arg0) {
                spinnerIntervalServer
                        .setEnabled(!spinnerIntervalServer.isEnabled());
                setAutoRetrieve(chckbxAutomaticallyRetrieve.isSelected());
            }
        });
        chckbxAutomaticallyRetrieve.setSelected(true);

        JPanel intervalPanelSever = new JPanel();
        automaticRequestPanel.add(intervalPanelSever);
        intervalPanelSever.setLayout(new GridLayout(1, 2, 5, 0));

        JLabel lblSecondsServer = new JLabel("Interval in seconds");
        lblSecondsServer.setToolTipText("Allowed range [1,900]");
        lblSecondsServer.setHorizontalAlignment(SwingConstants.RIGHT);
        intervalPanelSever.add(lblSecondsServer);
        spinnerIntervalServer.setEnabled(false);

        intervalPanelSever.add(spinnerIntervalServer);
        spinnerIntervalServer.setModel(new SpinnerNumberModel(2, 1, 900, 1));

        JPanel filterMediaPanel = new JPanel();
        automaticRequestPanel.add(filterMediaPanel);
        filterMediaPanel.setLayout(new GridLayout(1, 2, 5, 0));

        chckbxFilterMedia.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent arg0) {
                filterMedia(chckbxFilterMedia.isSelected(),
                        (String) comboBoxFilterMedia.getSelectedItem());
                comboBoxFilterMedia
                        .setEnabled(!comboBoxFilterMedia.isEnabled());
            }
        });
        chckbxFilterMedia.setHorizontalAlignment(SwingConstants.LEFT);
        chckbxFilterMedia.setSelected(true);
        filterMediaPanel.add(chckbxFilterMedia);
        comboBoxFilterMedia.setEnabled(false);

        comboBoxFilterMedia
                .setModel(new DefaultComboBoxModel<String>(new String[] {
                        "Abstract", "Electricity", "Water", "Gas", "Heat" }));
        comboBoxFilterMedia.setSelectedIndex(1);
        filterMediaPanel.add(comboBoxFilterMedia);

        JPanel filterClassPanel = new JPanel();
        automaticRequestPanel.add(filterClassPanel);
        filterClassPanel.setLayout(new GridLayout(1, 2, 5, 0));

        chckbxFilterClass.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                filterClass(chckbxFilterClass.isSelected(),
                        (String) comboBoxFilterClass.getSelectedItem());
                comboBoxFilterClass
                        .setEnabled(!comboBoxFilterClass.isEnabled());
            }
        });
        chckbxFilterClass.setHorizontalAlignment(SwingConstants.LEFT);
        filterClassPanel.add(chckbxFilterClass);

        comboBoxFilterClass.setModel(new DefaultComboBoxModel<String>(
                new String[] { "Data", "Register" }));
        comboBoxFilterClass.setSelectedIndex(1);
        filterClassPanel.add(comboBoxFilterClass);

        // rest settings panel
        JPanel restPanel = new JPanel();
        restPanel.setBorder(
                new TitledBorder(UIManager.getBorder("TitledBorder.border"),
                        "RESTful Webservice Settings", TitledBorder.LEADING,
                        TitledBorder.TOP, null, new Color(0, 0, 0)));
        GridBagConstraints gbc_restPanel = new GridBagConstraints();
        gbc_restPanel.anchor = GridBagConstraints.SOUTH;
        gbc_restPanel.fill = GridBagConstraints.BOTH;
        gbc_restPanel.gridx = 1;
        gbc_restPanel.gridy = 1;
        clientPanel.add(restPanel, gbc_restPanel);
        restPanel.setLayout(new GridLayout(4, 1, 0, 5));

        JLabel lblRestUrl = new JLabel("RESTful Webservice URL");
        lblRestUrl.setVerticalAlignment(SwingConstants.BOTTOM);
        restPanel.add(lblRestUrl);
        lblRestUrl.setHorizontalAlignment(SwingConstants.CENTER);

        textFieldRestUrl = new JTextField();
        restPanel.add(textFieldRestUrl);
        textFieldRestUrl
                .setText("http://localhost:8080/RestServerEMT/emt/cosem/ldevs");
        textFieldRestUrl.setColumns(30);

        JPanel intervalPanelRest = new JPanel();
        restPanel.add(intervalPanelRest);
        intervalPanelRest.setLayout(new GridLayout(1, 2, 5, 0));

        JLabel lblSecondsRest = new JLabel("Interval in seconds");
        lblSecondsRest.setToolTipText("Allowed range [4,900]");
        lblSecondsRest.setHorizontalAlignment(SwingConstants.RIGHT);
        intervalPanelRest.add(lblSecondsRest);

        final JSpinner spinnerIntervalRest = new JSpinner();
        spinnerIntervalRest.setModel(new SpinnerNumberModel(5, 4, 900, 1));
        intervalPanelRest.add(spinnerIntervalRest);

        JPanel restButtonPanel = new JPanel();
        restPanel.add(restButtonPanel);
        restButtonPanel.setLayout(new GridLayout(1, 2, 0, 0));

        btnConnectRest.setEnabled(false);
        btnConnectRest.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent arg0) {
                SpinnerModel numberModel = spinnerIntervalRest.getModel();
                setupRestClient(true, textFieldRestUrl.getText(),
                        ((SpinnerNumberModel) numberModel).getNumber()
                                .intValue());
                textFieldRestUrl.setEditable(false);
                spinnerIntervalRest.setEnabled(false);
                btnConnectRest.setEnabled(false);
                btnDisconnectRest.setEnabled(true);
            }
        });
        restButtonPanel.add(btnConnectRest);

        btnDisconnectRest.setEnabled(false);
        btnDisconnectRest.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                setupRestClient(false, "", 0);
                textFieldRestUrl.setEditable(true);
                spinnerIntervalRest.setEnabled(true);
                btnConnectRest.setEnabled(true);
                btnDisconnectRest.setEnabled(false);
            }
        });
        restButtonPanel.add(btnDisconnectRest);

    }

    /**
     * Sets the server settings text-fields editable.
     * 
     * @param b
     */
    public void setServerSettingsEditable(boolean b) {
        textFieldServerIp.setEditable(b);
        textFieldServerPort.setEditable(b);
        textFieldServerId.setEditable(b);
        textFieldClientId.setEditable(b);
        textFieldUsername.setEditable(b);
        passwordField.setEditable(b);
    }

    /**
     * Initializes the client, based on the user input on the GUI. Returns true
     * if successful, else false.
     * 
     * @return
     */
    public boolean initClient() {
        String clientID = textFieldClientId.getText();
        String serverID = textFieldServerId.getText();
        String serverIP = textFieldServerIp.getText();

        try { // check if IP is valid
            InetAddress addr = InetAddress.getByName(serverIP);
            textFieldServerIp.setText(addr.getHostName());
        } catch (UnknownHostException e) {
            // e.printStackTrace();
            textFieldServerIp.setText("Invalid IP address");
            return false;
        }

        int port = 0;
        try { // check if port is valid
            port = Integer.parseInt(textFieldServerPort.getText());
            if (port <= 0 || port > 65535)
                throw new NumberFormatException();
        } catch (NumberFormatException e) {
            // e.printStackTrace();
            textFieldServerPort.setText("Invalid Port");
            return false;
        }

        // init user authentication, if available
        String username = textFieldUsername.getText();
        String password = String.valueOf(passwordField.getPassword());

        // init client instance
        client = new Client();
        client.initialize(clientID, serverID, serverIP, port, username,
                password);

        // init interval
        setIntervalClient(
                ((SpinnerNumberModel) spinnerIntervalServer.getModel())
                        .getNumber().intValue());

        // init filter
        filterMedia(chckbxFilterMedia.isSelected(),
                (String) comboBoxFilterMedia.getSelectedItem());
        filterClass(chckbxFilterClass.isSelected(),
                (String) comboBoxFilterClass.getSelectedItem());

        // start timer (refreshes jlists)
        timer.start();

        return true;
    }

    /**
     * Sets the request interval in the client. Allowed range is [1,900]
     * seconds.
     * 
     * @param interval
     */
    private void setIntervalClient(int interval) {
        if (interval > 0 && interval <= 900 && client != null) {
            client.setInterval(interval * 1000);
        }
    }

    /**
     * Sets if the client automatically retrieves values from the server.
     * 
     * @param b
     */
    private void setAutoRetrieve(boolean b) {
        if (client != null) {
            client.setIsSendingAuto(b);
            if (b) {
                setIntervalClient(
                        ((SpinnerNumberModel) spinnerIntervalServer.getModel())
                                .getNumber().intValue());
            }
        }
    }

    /**
     * Adds a manual action request to the client. Currently, no parameters
     * supported via GUI. (Because there are currently no methods which have
     * params)
     * 
     * @param index
     */
    private void addManualActionRequest(short methodIndex) {
        InterfaceClass ic = objectJList.getSelectedValue();
        if (ic == null || methodIndex == 0 || client == null) { // no object
                                                                // selected
            System.out.println(
                    "Action request: no object in list selected, or invalid method index.");
        } else {
            ActionRequest request = new ActionRequest(
                    ic.getLogicalName().toString(), ic.getClassID().getVal(),
                    ic.getClassVersion().getVal(), methodIndex, null);
            client.addManualRequest(request);
            System.out.println("Action request: added to client.");
        }
    }

    /**
     * Adds a manual set request to the client. Currently, only supports
     * long/Unsigned64 values via the GUI. (Because handling differenz datatypes
     * via GUI is complicated)
     * 
     * @param value
     */
    private void addManualSetRequest(short index, String valueString) {
        InterfaceClass ic = objectJList.getSelectedValue();
        if (ic == null || index == 0 || client == null) { // no object selected
            System.out.println(
                    "Set request: No object in list selected, or invalid attribute index.");
        } else {
            try {
                long value = Long.parseLong(valueString);
                SML_CosemValue val = new SML_CosemValue(new Unsigned64(value));
                SetRequest request = new SetRequest(
                        ic.getLogicalName().toString(),
                        ic.getClassID().getVal(), ic.getClassVersion().getVal(),
                        new short[] { index }, new SML_CosemValue[] { val });
                client.addManualRequest(request);
                System.out.println("Set request: added to client.");
            } catch (NumberFormatException e) {
                System.out.println("Set request: invalid parameter.");
            }
        }
    }

    /**
     * Adds a manual get request to the client. Currently, only the selective
     * access to one attribute per request via GUI. (because handling multiple
     * indexes via GUI is complicated)
     * 
     * @param attributeIndex
     */
    private void addManualGetRequest(short attributeIndex) {
        InterfaceClass ic = objectJList.getSelectedValue();
        if (ic == null || client == null) { // no object selected
            System.out.println(
                    "Get request: No object in list selected, or invalid attribute index.");
        } else {
            GetRequest request = new GetRequest(ic.getLogicalName().toString(),
                    ic.getClassID().getVal(), ic.getClassVersion().getVal(),
                    new short[] { attributeIndex });
            client.addManualRequest(request);
            System.out.println("Get request: added to client.");
        }
    }

    /**
     * Manage the media filter depening on the user input.
     * 
     * @param b
     *            if active or inactive
     * @param medium
     *            medium to filter
     */
    private void filterMedia(boolean b, String medium) {
        if (b) { // filter active
            if (client != null && client.getFilter() != null) {
                if (medium.equals("Electricity")) {
                    client.getFilter().setFilterMedia(new String[] { "1" });
                }
                if (medium.equals("Abstract")) {
                    client.getFilter().setFilterMedia(new String[] { "0" });
                }
            }
        } else { // filter inactive
            if (client != null && client.getFilter() != null) {
                client.getFilter().setFilterMedia(new String[] {});
            }
        }
    }

    /**
     * Manage the class filter depening on the user input.
     * 
     * @param b
     *            if active or inactive
     * @param classID
     *            class id to filter
     */
    private void filterClass(boolean b, String classID) {
        if (b) { // filter active
            if (client != null && client.getFilter() != null) {

                if (classID.equals("Data")) {
                    client.getFilter().setFilterClass(new int[] { 1 });
                }
                if (classID.equals("Register")) {
                    client.getFilter().setFilterClass(new int[] { 3 });
                }
            }
        } else { // filter inactive
            if (client != null && client.getFilter() != null) {
                client.getFilter().setFilterClass(new int[] {});
            }
        }
    }

    /**
     * Set up the REST client with a destination address, or stop sending.
     * 
     * @param isSending
     *            true if active, else inactive
     * @param url
     *            REST server destination URL address
     */
    public void setupRestClient(boolean isSending, String url, int interval) {
        if (isSending) {
            if (url.length() > 1 && interval > 0 && interval < 900) {
                client.getRestClient().setServerURL(url);
                client.getRestClient().setInterval(interval * 1000); // s to ms
                client.getRestClient().setIsSending(isSending);
            } else {
                System.out.println(
                        "RestClient: parameter was invalid, sending not started!");
            }
        } else {
            client.getRestClient().setIsSending(isSending);
        }

    }

    /**
     * Disconnect client and stop all actions.
     */
    public void disconnect() {
        timer.stop();
        chartPanel.reset();
        client.interrupt();
        objectListModel.clear();
        attributeListModel.clear();
    }

    /**
     * Draws the diagram of the selected object from the list on the chart
     * panel.
     */
    public void drawDiagram() {
        InterfaceClass ic = objectJList.getSelectedValue();
        if (ic != null) {
            chartPanel.drawNewChart(ic);
        }
    }
}
