/**
 * Lookup Tables (Header+Implementation)
 * 
 * Copyright 2013 Fabian Schrodt, FSchrodt@gmx.de
 * 
 * This file is part of RNNPBlib.
 * 
 * RNNPBlib is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License version 3 as published by the Free Software Foundation.
 * 
 * RNNPBlib is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with RNNPBlib. If not, see http://www.gnu.org/licenses/.
 */

#include <iostream>
#include <math.h>

using namespace std;

#pragma once
#ifdef ENABLE_LOOKUP_TABLES

#define LUT_SIZE 1000000.0
#define LUT_SPAN 10.0		//-10 to 10

struct RNNPB_LUT
{
	double* sigmoidLUT;
	double* tanhLUT;
	double* gaussLUT;

	double* sigmoidDiffLUT;
	double* tanhDiffLUT;
	double* gaussDiffLUT;

	void deleteLUTs()
	{
		if(sigmoidLUT==NULL)
			return;

		delete[] sigmoidLUT;
		sigmoidLUT=NULL;
		delete[] tanhLUT;
		delete[] gaussLUT;
		delete[] sigmoidDiffLUT;
		delete[] tanhDiffLUT;
		delete[] gaussDiffLUT;
	}

	void prepareLUTs()
	{
		cout<<"PREPARING LUTS!\n";

		sigmoidLUT=new double[int(LUT_SIZE)];
		tanhLUT=new double[int(LUT_SIZE)];
		gaussLUT=new double[int(LUT_SIZE)];
		sigmoidDiffLUT=new double[int(LUT_SIZE)];
		tanhDiffLUT=new double[int(LUT_SIZE)];
		gaussDiffLUT=new double[int(LUT_SIZE)];

		for(int i=0;i<int(LUT_SIZE);i++)
		{
			double x = (double(i)/LUT_SIZE)*(2.0*LUT_SPAN)-LUT_SPAN;

			//sigmoid
			sigmoidLUT[i] = 1.0 / (1.0 + exp(- x));
			sigmoidDiffLUT[i] = 1.0 / (4.0 * pow(cosh(x / 2.0),2.0));

			//tanh
			tanhLUT[i] = tanh(x);
			tanhDiffLUT[i] = 1.0 / pow(cosh(x),2.0);

			//gaussian
			gaussLUT[i] = exp(-0.5*pow(x,2.0));
			gaussDiffLUT[i] = exp(-0.5*pow(x,2.0))*(-x);
		}
	}

	RNNPB_LUT()
	{
		prepareLUTs();
	}
	~RNNPB_LUT()
	{
		deleteLUTs();
	}
};

#endif
