/**
 * Datastructures for Trainingdata (Header+Implementation)
 * 
 * Copyright 2013 Fabian Schrodt, FSchrodt@gmx.de
 * 
 * This file is part of RNNPBlib.
 * 
 * RNNPBlib is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License version 3 as published by the Free Software Foundation.
 * 
 * RNNPBlib is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with RNNPBlib. If not, see http://www.gnu.org/licenses/.
 */

#pragma once

#include "RNNPB_Definitions.h"

#include "RNNPB_Helper.h"
#include "RNNPB_Vector.h"

#include <vector>
#include <deque>
using std::vector;
using std::deque;


#include <fstream>

#include <boost/archive/binary_oarchive.hpp>
#include <boost/archive/binary_iarchive.hpp>

/*
 * Container for one timeseries of trainingdata.
 * */
class RNNPB_TrainingDataContainer
{
	friend class boost::serialization::access;

	deque <RNNPB_Vector*> input;
	deque <RNNPB_Vector*> output_target;

	unsigned int input_dimension;
	unsigned int target_dimension;

public:

	unsigned int size()
	{
		return input.size();
	}

	/*
	 * Sets sample if not NULL;
	 * Returns address of input-sample.
	 * */
	RNNPB_Vector* inputSample(unsigned int position, RNNPB_Vector* set_input = NULL)
	{
		//resize
		if(position+1>size())
		{
			#ifdef ENABLE_DEBUG
				//cout<<"Warning: Inefficient Trainingdata access!\n";
			#endif
			input.resize(position+1);
		}

		if(input[position]==NULL)
			input[position]=new RNNPB_Vector(input_dimension);

		if(set_input!=NULL)
			*(input[position])=*set_input;

		return input[position];
	}

	/*
	 * Sets sample if not NULL;
	 * Returns address of target-sample.
	 * */
	RNNPB_Vector* targetSample(unsigned int position, RNNPB_Vector* set_output_target = NULL)
	{
		if(position+1>output_target.size())
		{
			#ifdef ENABLE_DEBUG
				//cout<<"Warning: Inefficient Trainingdata access!\n";
			#endif
			output_target.resize(position+1);
		}

		if(output_target[position]==NULL)
			output_target[position]=new RNNPB_Vector(target_dimension);

		if(set_output_target!=NULL)
			*(output_target[position])=*set_output_target;

		return output_target[position];
	}

	void deleteSample(unsigned int position)
	{
		input.erase(input.begin()+position);
		output_target.erase(output_target.begin()+position);
	}

	void clear()
	{
		for(unsigned int i=0;i<input.size();i++)
			delete input[i];
		for(unsigned int i=0;i<output_target.size();i++)
			delete output_target[i];

		input.resize(0);
		output_target.resize(0);
	}

	RNNPB_TrainingDataContainer(unsigned int set_input_dimension, unsigned int set_target_dimension)
	{
		input_dimension=set_input_dimension;
		target_dimension=set_target_dimension;
	}

	~RNNPB_TrainingDataContainer()
	{
		for(unsigned int i=0;i<input.size();i++)
			delete input[i];
		for(unsigned int i=0;i<output_target.size();i++)
			delete output_target[i];
	}
};

/*
 * Dynamic container for timeseries of all behaviors and epochs.
 * */
struct RNNPB_TrainingDataPerBehaviourContainer
{
    friend class boost::serialization::access;

	deque <deque <RNNPB_TrainingDataContainer*>> data;	//data[behaviour][epoch]
public:

	// These values must be given by the trainer and are fixed for all trainingdata:
	unsigned int input_dimension;
	unsigned int target_dimension;

	unsigned int nrOfEpochs(unsigned int behaviour)
	{
		return data[behaviour].size();
	}

	unsigned int nrOfBehaviours()
	{
		return data.size();
	}

	/*
	 * This fiteres inconsistend training data for non-recurrent neural networks (see documentation).
	 * */
	unsigned int filterTrainingData(double deletion_threshold, int behaviour = -1)
	{
		unsigned int filteredAll=0;
		for(unsigned int b=0;b<data.size();b++)
		{
			if(behaviour!=-1)
				b=behaviour;			

			unsigned int filtered=0;

			for(unsigned int e1 = 0;e1<data[b].size();e1++)
			{
				for(unsigned int s1 = 0;s1<data[b][e1]->size();s1++)
				{
					for(unsigned int e2 = e1+1;e2<data[b].size();e2++)
					{
						for(unsigned int s2 = s1+1;s2<data[b][e2]->size();s2++)
						{
							double in_dist = RNNPB_Vector::euclideanDistance(data[b][e1]->inputSample(s1), data[b][e2]->inputSample(s2));
							double tar_dist = RNNPB_Vector::euclideanDistance(data[b][e1]->targetSample(s1), data[b][e2]->targetSample(s2));

							if(tar_dist / in_dist > deletion_threshold)
							{
								deleteTrainingSample(s2, b, e2);
								cout<<"[(e"<<e1<<", s"<<s1<<"),(e"<<e2<<", s"<<s2<<")]"<<endl;
								filtered++;
								s2--;
							}
						}
					}
				}
			}
			filteredAll+=filtered;

			cout<<"RNNPB: Conflicting data for behaviour "<<b<<": ";

			if(filtered==0)
				cout<<" none.\n";
			else
				cout<<filtered<<endl;

			if(behaviour!=-1)
				break;
		}

		return filteredAll;
	}

	RNNPB_TrainingDataContainer* trainingEpoch(unsigned int behaviour, unsigned int epoch, RNNPB_TrainingDataContainer* set_to = NULL)
	{
		if(behaviour+1>data.size())	//resize
		{
			data.resize(behaviour+1);
		}

		unsigned int oldsize = data[behaviour].size();
		if(epoch+1>oldsize)	//resize
		{
			data[behaviour].resize(epoch+1);

			for(unsigned int e=oldsize;e<=epoch;e++)
			{
				data[behaviour][e]=new RNNPB_TrainingDataContainer(input_dimension, target_dimension);

				if(set_to!=NULL)
				{
					for(unsigned int i=0;i<set_to->size();i++)
					{
						data[behaviour][e]->inputSample(i, set_to->inputSample(i));		//copies data!
						data[behaviour][e]->targetSample(i, set_to->targetSample(i));		//copies data!
					}
				}
			}
		}

		return data[behaviour][epoch];
	}

	RNNPB_Vector* trainingInput(unsigned int iteration, unsigned int behaviour, unsigned int epoch, RNNPB_Vector* set_to = NULL)
	{
		return trainingEpoch(behaviour, epoch)->inputSample(iteration, set_to);
	}

	RNNPB_Vector* trainingTarget(unsigned int iteration, unsigned int behaviour, unsigned int epoch, RNNPB_Vector* set_to = NULL)
	{
		return trainingEpoch(behaviour, epoch)->targetSample(iteration, set_to);
	}

	void deleteTrainingSample(unsigned int iteration, unsigned int behaviour, unsigned int epoch)
	{
		if(behaviour < data.size() && epoch < data[behaviour].size())
			data[behaviour][epoch]->deleteSample(iteration);
		else
			cout<<"Error on training sample deletion at behaviour "<<behaviour<<" and epoch "<<epoch<<endl;
	}

	void clearTrainingEpoch(unsigned int behaviour, unsigned int epoch)
	{
		if(behaviour < data.size() && epoch < data[behaviour].size())
		{
			//data[behaviour].erase(data[behaviour].begin()+epoch);	//shifts indices!!!
			data[behaviour][epoch]->clear();
		}
		//else
			//cout<<"Error on training epoch deletion at behaviour "<<behaviour<<" and epoch "<<epoch<<endl;
	}

	void deleteTrainingEpoch(unsigned int behaviour, unsigned int epoch)
	{
		if(behaviour < data.size() && epoch < data[behaviour].size())
		{
			data[behaviour].erase(data[behaviour].begin()+epoch);	//shifts indices!!!
			//data[behaviour][epoch]->clear();
		}
		//else
			//cout<<"Error on training epoch deletion at behaviour "<<behaviour<<" and epoch "<<epoch<<endl;
	}

	// When the class Archive corresponds to an output archive, the
	// & operator is defined similar to <<.  Likewise, when the class Archive
	// is a type of input archive the & operator is defined similar to >>.
	template<class Archive>	void serialize(Archive & ar, const unsigned int version)
	{
		unsigned int behaviours, epochs, iterations;

		behaviours=data.size();

		bool load=false;
		if(behaviours==0)
			load=true;

		ar & behaviours;

		for(unsigned int a=0;a<behaviours;a++)	//# behaviour
		{
			if(!load)
				epochs=data[a].size();
			ar & epochs;

			for(unsigned int b=0;b<epochs;b++)	//# epoch
			{
				if(!load)
					iterations = data[a][b]->size();
				ar & iterations;

				for(unsigned int c=0;c<iterations;c++)	//# iterations
				{
					for(unsigned int d=0;d<input_dimension;d++)		//in-dimension
					{
						ar & (*trainingInput(c,a,b))[d];
					}
					for(unsigned int d=0;d<target_dimension;d++)	//out-dimension
					{
						ar & (*trainingTarget(c,a,b))[d];
					}
				}
			}
		}
	}

	RNNPB_TrainingDataPerBehaviourContainer
		(
				unsigned int set_input_dimension,
				unsigned int set_target_dimension
		)
	{
		input_dimension=set_input_dimension;
		target_dimension=set_target_dimension;
	}

	~RNNPB_TrainingDataPerBehaviourContainer()
	{
		for(unsigned int i=0;i<data.size();i++)
		{
			for(unsigned int j=0;j<data[i].size();j++)
			{
				clearTrainingEpoch(i, j);
			}
		}
	}
};
