/**
 * Vector-type of this framework (Header+Implementation)
 * 
 * Copyright 2013 Fabian Schrodt, FSchrodt@gmx.de
 * 
 * This file is part of RNNPBlib.
 * 
 * RNNPBlib is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License version 3 as published by the Free Software Foundation.
 * 
 * RNNPBlib is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with RNNPBlib. If not, see http://www.gnu.org/licenses/.
 */

#pragma once

#include "RNNPB_Definitions.h"
#include "RNNPB_Helper.h"

#include <vector>
#define _USE_MATH_DEFINES
#include <math.h>
#include <cmath>
#include <stdlib.h>
#include <string.h>
#include <iostream>
using namespace std;

struct RNNPB_Vector
{
	const unsigned int size;

	double* data;

	inline double& operator[](int index) const
	{
	    return data[index];
	}

	inline RNNPB_Vector& operator=( const RNNPB_Vector& set )
	{
		#ifdef ENABLE_DEBUG
		if(size!=set.size)
		{
			cout<<"Wrong assignment size: "<<size<<"!="<<set.size<<"\n";
			return *this;
		}
		#endif

		memcpy((void*)data, (void*)set.data, sizeof(double)*size);

		return *this;
	}

	inline RNNPB_Vector& operator=( double* set )
	{
		memcpy((void*)data, (void*)set, sizeof(double)*size);

		return *this;
	}

	/*inline RNNPB_Vector& operator=( const double& set )
	{
		for(unsigned int i=0;i<size;i++)
			data[i]=set;

		return *this;
	}*/

	inline void clear()
	{
		for(unsigned int i=0;i<size;i++)
			data[i]=0;
	}


	/*
	 * Copy-elementwise-product
	 * */
	inline RNNPB_Vector& operator*=( const RNNPB_Vector& rhs )
	{
		#ifdef ENABLE_DEBUG
		if(size!=rhs.size)
		{
			cout<<"Wrong assignment size: "<<size<<"!="<<rhs.size<<"\n";
			return *this;
		}
		#endif

		for(unsigned int i=0;i<size;i++)
			data[i]=data[i]*rhs.data[i];

		return *this;
	}

	/*
	 * Copy-elemenwise-division
	 * */
	inline RNNPB_Vector& operator/=( const RNNPB_Vector& rhs )
	{
		#ifdef ENABLE_DEBUG
		if(size!=rhs.size)
		{
			cout<<"Wrong assignment size: "<<size<<"!="<<rhs.size<<"\n";
			return *this;
		}
		#endif

		for(unsigned int i=0;i<size;i++)
			data[i]=data[i]/rhs.data[i];

		return *this;
	}

	/*
	 * Copy-minus
	 * */
	inline RNNPB_Vector& operator-=( const RNNPB_Vector& rhs )
	{
		#ifdef ENABLE_DEBUG
		if(size!=rhs.size)
		{
			cout<<"Wrong assignment size: "<<size<<"!="<<rhs.size<<"\n";
			return *this;
		}
		#endif

		for(unsigned int i=0;i<size;i++)
			data[i]=data[i]-rhs.data[i];

		return *this;
	}

	/*
	 * Copy-plus
	 * */
	inline RNNPB_Vector& operator+=( const RNNPB_Vector& rhs )
	{
		#ifdef ENABLE_DEBUG
		if(size!=rhs.size)
		{
			cout<<"Wrong assignment size: "<<size<<"!="<<rhs.size<<"\n";
			return *this;
		}
		#endif

		for(unsigned int i=0;i<size;i++)
			data[i]=data[i]+rhs.data[i];

		return *this;
	}

	/*
	 * Copy-product
	 * */
	inline RNNPB_Vector& operator*=( const double& rhs )
	{
		for(unsigned int i=0;i<size;i++)
			data[i]=data[i]*rhs;

		return *this;
	}

	/*
	 * Copy-division
	 * */
	inline RNNPB_Vector& operator/=( const double& rhs )
	{
		for(unsigned int i=0;i<size;i++)
			data[i]=data[i]/rhs;

		return *this;
	}

	void print()
	{
		std::cout<<"[";
		for(unsigned int i=0;i<size-1;i++)
		{
			std::cout<<data[i]<<", ";
		}
		std::cout<<data[size-1]<<"]\n";
	}

	double length()
	{
		double ret=0;
		for(unsigned int i=0;i<size;i++)
			ret+=pow(data[i],2.0);

		#ifdef ENABLE_DEBUG
			if(sqrt(ret)!=sqrt(ret) || sqrt(ret)>10.0 || sqrt(ret)<0) //sqrt(ret)>1.0e30)
			{
				cout<<"WARNING! Strange RNNPB_Vector value: ";
				print();
			}
		#endif

		return sqrt(ret);
	}

	double lengthLin()
	{
		double ret=0;
		for(unsigned int i=0;i<size;i++)
			ret+=std::abs(data[i]);

		return ret;
	}

	static RNNPB_Vector abs(const RNNPB_Vector & vec)
	{
		RNNPB_Vector ret(vec.size);

		for(unsigned int i=0;i<vec.size;i++)
			ret[i]=std::abs(vec.data[i]);

		return ret;
	}

	static double euclideanDistance(RNNPB_Vector* a, RNNPB_Vector* b)
	{
		#ifdef ENABLE_DEBUG
		if(a->size!=b->size)
			cout<<"Dimension Error: "<<a->size<<"!="<<b->size<<"\n";
		#endif

		double ret = 0;
		for(unsigned int i=0;i<a->size;i++)
		{
			ret += pow((*a)[i] - (*b)[i],2.0);
		}

		return sqrt(ret);
	}

	/*static double euclideanDistanceSignum(RNNPB_Vector* a, RNNPB_Vector* b)	//test
	{
		#ifdef ENABLE_DEBUG
		if(a->size!=b->size)
			cout<<"Dimension Error: "<<a->size<<"!="<<b->size<<"\n";
		#endif

		double ret = 0;
		double sign = 0;
		for(unsigned int i=0;i<a->size;i++)
		{
			sign += (*a)[i] - (*b)[i];
			ret += pow((*a)[i] - (*b)[i], 2.0);
		}

		return sqrt(ret)*RNNPB_Helper::signum(sign);
	}*/

	static double halfSquaredError(RNNPB_Vector* a, RNNPB_Vector* b)
	{
		#ifdef ENABLE_DEBUG
		if(a->size!=b->size)
			cout<<"Dimension Error: "<<a->size<<"!="<<b->size<<"\n";
		#endif

		double ret = 0;
		for(unsigned int i=0;i<a->size;i++)
		{
			ret += pow((*a)[i] - (*b)[i],2.0);
		}

		return 0.5*ret;
	}

	/*static double halfSquaredErrorSignum(RNNPB_Vector* a, RNNPB_Vector* b) //test
	{
		#ifdef ENABLE_DEBUG
		if(a->size!=b->size)
			cout<<"Dimension Error: "<<a->size<<"!="<<b->size<<"\n";
		#endif

		double ret = 0;
		double sign = 0;
		for(unsigned int i=0;i<a->size;i++)
		{
			sign += (*a)[i] - (*b)[i];
			ret += pow((*a)[i] - (*b)[i], 2.0);
		}

		return 0.5*ret*RNNPB_Helper::signum(sign);
	}*/

	static double scalar_prod(RNNPB_Vector veca, RNNPB_Vector vecb)
	{
		double tmp=0;
		for(unsigned int i=0;i<veca.size;i++)
		{
			tmp += veca[i] * vecb[i];
		}
		return tmp;
	}

	static double angleBetween(RNNPB_Vector veca, RNNPB_Vector vecb)
	{
		//return acos(norm_cos_angle(veca, vecb, veca.size()));

		return acos(scalar_prod(veca, vecb) / (veca.length() * vecb.length()));
	}

	bool isCorrupted()
	{
		for(unsigned int i=0;i<size;i++)
		{
			if(data[i]!=data[i])
				return true;
		}
		return false;
	}

	/*
	 * Copy constructor
	 */
	RNNPB_Vector(const RNNPB_Vector &set) : size(set.size)
	{
		#ifdef WIN32
			data = (double*) malloc(sizeof(double)*(size+1));	//i dont know why windows wants that...
		#else
			data = (double*) malloc(sizeof(double)*(size));
		#endif

		memcpy((void*)data, (void*)set.data, sizeof(double)*size);
	}

	RNNPB_Vector(unsigned int set_size) : size(set_size)
	{
		#ifdef WIN32
			data = (double*) malloc(sizeof(double)*(size+1));	//i dont know why windows wants that...
		#else
			data = (double*) malloc(sizeof(double)*(size));
		#endif
	}

	/*
	 * This constructor uses preallocated memory ("reference constructor")! Be careful with deletes!
	 * */
	RNNPB_Vector(unsigned int set_size, double* set_data) : size(set_size)
	{
		data=set_data;
	}

	~RNNPB_Vector()
	{
		free(data);
	}
};

inline const RNNPB_Vector operator* (const RNNPB_Vector & lhs, const RNNPB_Vector & rhs)
{
	RNNPB_Vector ret(lhs.size);

	#ifdef ENABLE_DEBUG
	if(lhs.size!=rhs.size)
	{
		cout<<"Dimension Error: "<<lhs.size<<"!="<<rhs.size<<"\n";
		//__asm{int 3};
		return ret;
	}
	#endif

	for(unsigned int i=0;i<lhs.size;i++)
		ret[i]=lhs[i] * rhs[i];

	return ret;
}

inline const RNNPB_Vector operator* (const RNNPB_Vector & lhs, const double & rhs)
{
	RNNPB_Vector ret(lhs.size);

	for(unsigned int i=0;i<lhs.size;i++)
		ret[i]=lhs[i] * rhs;

	return ret;
}

inline const RNNPB_Vector operator+ (const RNNPB_Vector& lhs, const RNNPB_Vector& rhs)
{
	RNNPB_Vector ret(lhs.size);

	#ifdef ENABLE_DEBUG
	if(lhs.size!=rhs.size)
	{
		cout<<"Dimension Error: "<<lhs.size<<"!="<<rhs.size<<"\n";
		//__asm{int 3};
		return ret;
	}
	#endif

	for(unsigned int i=0;i<lhs.size;i++)
		ret[i]=lhs[i] + rhs[i];

	return ret;
}

inline const RNNPB_Vector operator- (const RNNPB_Vector& lhs, const RNNPB_Vector& rhs)
{
	RNNPB_Vector ret(lhs.size);

	#ifdef ENABLE_DEBUG
	if(lhs.size!=rhs.size)
	{
		cout<<"Dimension Error: "<<lhs.size<<"!="<<rhs.size<<"\n";
		//__asm{int 3};
		return ret;
	}
	#endif

	for(unsigned int i=0;i<lhs.size;i++)
		ret[i]=lhs[i] - rhs[i];

	return ret;
}
