/**
 * Weights (Header)
 * 
 * Copyright 2013 Fabian Schrodt, FSchrodt@gmx.de
 * 
 * This file is part of RNNPBlib.
 * 
 * RNNPBlib is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License version 3 as published by the Free Software Foundation.
 * 
 * RNNPBlib is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with RNNPBlib. If not, see http://www.gnu.org/licenses/.
 */

#pragma once

#include "RNNPB_Definitions.h"

#include <deque>
using std::deque;

class RNNPB_Neuron;
class RNNPB_NetworkContainer;

enum WeightType
{
	Normal,
	SecondOrder,
	Modulated,
	SecondOrderCV,
	ModulatedCV,
	ParametricBiasCV,
	TapDelayLine
};

class RNNPB_Weight
{
	friend class RNNPB_Neuron;
	friend class RNNPB_NeuronLayer;
	friend class RNNPB_Container;
	friend class RNNPB_CVTrainer;
	friend class RNNPB_AssemblyTools;
	friend class RNNPB_NetworkContainer;

	RNNPB_NetworkContainer* network;

	int parsed;

protected:

	virtual void subscribeNetwork(RNNPB_NetworkContainer* set_network, int run);

	WeightType weightType;

	RNNPB_Neuron* from;
	RNNPB_Neuron* to;

	double learningRate;
	double momentum;
	double weightDecay;

	double weight;
	double weightUpdate;

	bool constant;

	#ifdef ENABLE_SEPARATE_ANNEALING
	double maxWeightUpdate;
	double minWeightUpdate;
	#endif

	bool deltaErrorTimesActivationSumUptodate;
	double deltaErrorTimesActivationSum;

	virtual void iterate();

	virtual void clear();

	virtual double getOutput();

	virtual double getWeightedDeltaError(RNNPB_Neuron* caller);
	
	virtual void learn();

	virtual void cumulateWeightUpdates();

	virtual double getWeight();

public:

	RNNPB_Weight(RNNPB_Neuron* set_from, RNNPB_Neuron* set_to, WeightType set_type);
};



/*
 * Derived classes / weight types:
 * */


class RNNPB_NormalWeight : public RNNPB_Weight
{
public:

	RNNPB_NormalWeight(RNNPB_Neuron* set_from, RNNPB_Neuron* set_to);
};


class RNNPB_SecondOrderWeight : public RNNPB_Weight
{
	friend class RNNPB_NeuronLayer;

protected:

	double getWeight();

	void learn();

	void subscribeNetwork(RNNPB_NetworkContainer* set_network, int);

	double getWeightedDeltaError(RNNPB_Neuron* caller);

	RNNPB_Neuron* injection;

public:

	RNNPB_SecondOrderWeight(RNNPB_Neuron* set_from, RNNPB_Neuron* set_to, RNNPB_Neuron* set_injection);
};
class RNNPB_SecondOrderWeightCV : public RNNPB_SecondOrderWeight
{
	friend class RNNPB_NeuronLayer;

	void learn();

	//void clear();

public:

	RNNPB_SecondOrderWeightCV(RNNPB_Neuron* set_from, RNNPB_Neuron* set_to, RNNPB_Neuron* set_injection);
};



class RNNPB_ModulatedWeight : public RNNPB_SecondOrderWeight
{
	friend class RNNPB_NeuronLayer;

protected:

	double pb_weight;

	double getWeight();

	double getWeightedDeltaError(RNNPB_Neuron* caller);

public:

	RNNPB_ModulatedWeight(RNNPB_Neuron* set_from, RNNPB_Neuron* set_to, RNNPB_Neuron* set_injection);
};

class RNNPB_ModulatedWeightCV : public RNNPB_ModulatedWeight
{
	friend class RNNPB_NeuronLayer;

	void learn();

	//void clear();

public:

	RNNPB_ModulatedWeightCV(RNNPB_Neuron* set_from, RNNPB_Neuron* set_to, RNNPB_Neuron* set_injection);
};



class RNNPB_ParametricBiasWeightCV : public RNNPB_Weight
{
	friend class RNNPB_NeuronLayer;

	//void clear();

public:
	RNNPB_ParametricBiasWeightCV(RNNPB_Neuron* set_from, RNNPB_Neuron* set_to);
};


/*
 * Time delayed weight / recurrency. Can be set to exponentially smooth its input history (see documentation).
 * */
class RNNPB_TapDelayLineWeight : public RNNPB_Weight
{
	double outputDelayBuffer;
	double errorDelayBuffer;
	double decay;

	#ifdef ENABLE_BPTT
		//deque <double> outputDelayBufferBuffer;
		//deque <double> errorDelayBufferBuffer;
		double productSumBuffer;
		double deltaErrorSumBuffer;
		unsigned int iteration;
	#endif

	void cumulateWeightUpdates();

	double getWeightedDeltaError(RNNPB_Neuron* caller);

	void iterate();

	void clear();

	double getOutput();

public:

	RNNPB_TapDelayLineWeight(RNNPB_Neuron* set_from, RNNPB_Neuron* set_to, int set_delaySteps);
};






