/**
 * SimulationManager implementation file containing all its public and mayor functions (Implementation)
 * 
 * Copyright 2013 Fabian Schrodt, FSchrodt@gmx.de
 * 
 * This file is part of SimulationManager.
 * 
 * SimulationManager is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License version 3 as published by the Free Software Foundation.
 * 
 * SimulationManager is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with SimulationManager. If not, see http://www.gnu.org/licenses/.
 */

#include "SimulationManager.h"

#include "Simulation.h"
#include "SimulationCommand.h"

#include <fstream>
#include <iostream>
#include <vector>
#include <deque>
#include <string.h>
#include <stdio.h>
#include <dlfcn.h>
#include <stdlib.h>
#include <sys/resource.h>
#include <signal.h>
#include <errno.h>

using namespace std;

#define MAX_SIMS 40	//Achtung bei Abhaengigkeiten zwischen Simulationen und kleinen werten!
//TODO: Blockierendes und nicht-blockierendes StartSim-Kommando. Block: Einordnung in Warteschlange, wenn alle IDs belegt
//TODO: Verzögerte Antwort bei StopSimulation
#define MAX_BUFFER 10000


bool SimulationManager::validID(unsigned int simID)
{
	if(simID >= simulations.size())
		return false;

	if(simulations[simID] == NULL || simulations[simID]->shutDown)
		return false;

	return true;
}

/*
	Adds a new XML Simulation to the Manager
	Returns simulation-ID if successfull
	Returns -1 if full
	Returns -2 if error on child process creation
*/
int SimulationManager::addSimulation(SimulationInfo* set_simInfo)
{
	//TODO: Security check!? (How?)

	int ret = -1;

	//Wieder frei gewordene IDs belegen wenn alle IDs belegt wurden
	while( ret==-1 )
	{
		for(unsigned int i=0;i<simulations.size();i++)		//prüfe sims, wähle ersten freien slot
		{
			//check if simulation has finished
			if(simulations[i]==NULL)
			{
				if(ret==-1)
				{
					ret=i;
					break;
				}
			}
			//Attention: If an application attempts to use a thread ID whose lifetime has ended, the behavior is undefined.
			//so (pthread_kill(simulations[i]->sensorThread, 0)==ESRCH) or (!pthread_tryjoin_np(simulations[i]->sensorThread, NULL)) does not work!
			else if(pthread_mutex_trylock(simulations[i]->deleteMeLock)==0)	//if sensor thread finished
			{
				deleteSimulation(i);

				if(ret==-1)
				{
					ret=i;
					break;
				}
			}
		}

		if(ret==-1)	//kein freier slot gefunden
		{
			if(simulations.size() < MAX_SIMS)	//erweitere slots
			{
				ret=simulations.size();
				break;
			}

			//oder warte

			cout<<"MAN: Maximum number of Simulations reached. Waiting for a while!\n";

			pthread_mutex_unlock(man_lock);	//TODO: DON'T UNLOCK!?
			usleep(100000);
			pthread_mutex_lock(man_lock);

			ret=-1;	//neue suche

			//return -1;
		}
	}

	char output_pipename[100];
	snprintf(output_pipename, 100, "pipe-to-sim-%d", ret);
	char input_pipename[100];
	snprintf(input_pipename, 100, "pipe-to-man-%d", ret);


	char cmd[512];
	snprintf(cmd, 512, "%s %s %s &", set_simInfo->simName, input_pipename, output_pipename);


	//start processcontroller, TODO: correct buffer size
	if(ret == (int) simulations.size())
	{
		simulations.push_back(new PipeServerSimulation(ret,10000,output_pipename,input_pipename,cmd));
	}
	else
	{
		simulations[ret]=new PipeServerSimulation(ret,10000,output_pipename,input_pipename,cmd);
	}

	//START PROCESS
	int returncode=system(cmd);

	//"If command is not a null pointer, the value returned depends on the system and library implementations, but it is generally expected to be the status code returned by the called command, if supported."
	if(returncode==0 || returncode==-1)
	{
		cout<<"MAN: "<<"Process "<<set_simInfo->simName<<" started (simID: "<<ret<<")!\n";
	}
	else
	{
		cout<<"MAN: "<<"Could not start simulation '"<<set_simInfo->simName<<"'. Returncode: "<<returncode<<"\n";

		simulations.erase(simulations.begin()+ret);

		return -2;
	}

	activeSimulations++;

	return ret;
}


void SimulationManager::deleteSimulation(unsigned int simID)
{
	if(simulations[simID]==NULL)
		return;

	//delete simulationcontroller
	delete simulations[simID];
	simulations[simID]=NULL;
	activeSimulations--;

	//cout<<"MAN: "<<"Simulation "<<simID<<" was shut down...\n";
}

void SimulationManager::loadPlugins()
{
	ifstream input("../plugins.txt");

	char soname[100];

	input.getline(soname, 100);

	while(!input.eof())
	{
		if(soname[0]!='#')
			loadPlugin(soname);

		input.getline(soname, 100);
	}
}


/*
 * Public Functions
 * */

PluginInterface* SimulationManager::loadPlugin(const char* soname)
{
	//flush errors
	char *error = dlerror();

	cout<<"MAN: "<<"SimulationManager is loading plugin '"<<soname<<"'...\n";

	void* handle=dlopen(soname, RTLD_NOW | RTLD_LOCAL);
	error = dlerror();

	void* (*function)(void*) = (void* (*)(void*))dlsym(handle, "createPlugin");

	if(handle!=NULL && function != NULL && error == NULL)
	{
		PluginInterface* tmp = (PluginInterface*) function((void*) this);
		pthread_mutex_lock(man_lock);
		plugins[tmp]=handle;

		//start server
		pthread_create(tmp->server_thread,NULL,tmp->runThreadEntry,(void*) tmp);
		pthread_mutex_unlock(man_lock);

		cout<<"MAN: "<<"Run-thread was started.\n";

		return tmp;
	}
	else
	{
		cout<<"MAN: "<<" Error: Plugin could not be loaded. "<<"error: "<<error<<". so-address: "<<handle<<". function entry: "<<function<<"."<<endl;

		return NULL;
	}
}

int SimulationManager::unloadPlugin(PluginInterface* plugin)
{
	pthread_mutex_lock(man_lock);

	if(plugins.find(plugin) != plugins.end())
	{
		if(plugin->isRunning())
		{
			plugin->exit();
		}

		dlclose(plugins[plugin]);

		plugins.erase(plugins.find(plugin));

		delete plugin;
	}
	else
	{
		cout<<"MAN: Error: Plugin is not controlled by SimulationManager\n";
		return -1;
	}

	pthread_mutex_unlock(man_lock);
	return 0;
}


int SimulationManager::setSensorHandler(SensorHandlerInterface* set_sensorHandler, unsigned int simID)
{
	pthread_mutex_lock(man_lock);
	if(!validID(simID))
	{
		cout<<"MAN: "<<"Setting SensorHandler failed: Invalid simID!\n";
		pthread_mutex_unlock(man_lock);
		return -1;
	}

	//abarbeitung des sensorreaders stoppen:
	pthread_mutex_lock(simulations[simID]->sensorLock);
	simulations[simID]->sensorHandler=set_sensorHandler;
	pthread_mutex_unlock(simulations[simID]->sensorLock);

	pthread_mutex_unlock(man_lock);

	return 0;
}

/*
 * TODO: Kommandos Manager-intern nicht kopieren.
 * TODO: Besseres Locking pro Simulation.
 * */
SimulationCommand SimulationManager::applyCommand(SimulationCommand* cmd) //TODO: Strainge deadlock on multiple sims and plugins sometimes!?
{
	pthread_mutex_lock(man_lock);

	/*cout<<"command: "<<cmd->getMessageType()<<endl;
	cout<<"simID: "<<cmd->getSimulationID()<<endl<<endl;*/

	SimulationCommand ret;
	*(ret.getHeader()) = *(cmd->getHeader());

	unsigned int simID = cmd->getSimulationID();

	if( !validID(simID) && !cmd->getMessageType()==StartSimulation && !cmd->getMessageType()==StopSimulationNow )
	{
		ret.setMessageERR(1);

		pthread_mutex_unlock(man_lock);
		return ret;
	}

	/*
	 * TODO: SWITCH CASE!
	 * */

	if(cmd->getMessageType()==StartSimulation)
	{
		int mysimID = addSimulation(cmd->getBody()->dat<SimulationInfo>());

		if(mysimID>=0)
		{
			ret.setMessageACK(cmd->getAdditionalInfo());
			ret.getHeader()->dat<CommandHeader>()->simulationID = (int) mysimID;

			//fist command gets PID of the simulation:
			pthread_mutex_lock(simulations[mysimID]->queueLock);
			simulations[mysimID]->commandQueue.push_back(*cmd);
			pthread_mutex_unlock(simulations[mysimID]->emptyQueueLock);
			pthread_mutex_unlock(simulations[mysimID]->queueLock);
		}
		else if(mysimID==-1)
		{
			ret.setMessageERR(2);
		}
		else if(mysimID==-2)
		{
			ret.setMessageERR(3);
		}
	}
	else if(cmd->getMessageType()==StopSimulation)
	{
		//send terminate signal to simulation
		pthread_mutex_lock(simulations[simID]->queueLock);
		simulations[simID]->commandQueue.push_back(*cmd);
		pthread_mutex_unlock(simulations[simID]->emptyQueueLock);
		pthread_mutex_unlock(simulations[simID]->queueLock);

		simulations[simID]->shutDown=true;	//do not accept commands anymore

		//CHECK: DOES ALWAYS RETURN TRUE! Simulations will be deleted (finally) at next search for a free simulation slot!
		ret.setMessageACK(cmd->getAdditionalInfo());
	}
	else if(cmd->getMessageType()==StopSimulationNow)
	{
		if(simID>=simulations.size())	//invalid
		{
			ret.setMessageERR(6);
		}
		else if(simulations[simID]==NULL)	//already deleted
		{
			ret.setMessageACK(cmd->getAdditionalInfo());
		}
		else
		{
			//send terminate signal to simulation
			cmd->getHeader()->dat<CommandHeader>()->messageType=StopSimulation;
			pthread_mutex_lock(simulations[simID]->queueLock);
			simulations[simID]->commandQueue.push_back(*cmd);
			pthread_mutex_unlock(simulations[simID]->emptyQueueLock);
			pthread_mutex_unlock(simulations[simID]->queueLock);

			simulations[simID]->shutDown=true;	//do not accept commands anymore

			//wait for shut down...
			ret.setMessageACK(waitForSimulationToFinish(simID, cmd->getAdditionalInfo()));
		}
	}
	else if(cmd->getMessageType()==SetAction || cmd->getMessageType()==Step || cmd->getMessageType()==GetSensors)
	{
		pthread_mutex_lock(simulations[simID]->queueLock);

		if(cmd->getMessageType()==GetSensors && simulations[simID]->sensorHandler==NULL)
		{
			//avoid null-pointer exception when no sensorHandler is set
			ret.setMessageERR(5);
		}
		else //put command to queue of the simulation-thread
		{
			//Avoid buffer-overflow
			while(simulations[simID]->commandQueue.size() >= MAX_BUFFER)
			{
				cout<<"MAN: "<<"Manager avoided buffer overflow. Maybe there is some lag in piping.\n";

				pthread_mutex_unlock(simulations[simID]->queueLock);
				pthread_mutex_unlock(man_lock);	//CHECK: Maybe this is not absolutely consistent if some other thread wants to add a command to the queue of the same simulation...
				usleep(10000);	//TODO
				pthread_mutex_lock(man_lock);
				pthread_mutex_lock(simulations[simID]->queueLock);
			}

			simulations[simID]->commandQueue.push_back(*cmd);
			pthread_mutex_unlock(simulations[simID]->emptyQueueLock);

			ret.setMessageACK(cmd->getAdditionalInfo());
		}

		pthread_mutex_unlock(simulations[simID]->queueLock);
	}
	else if(cmd->getMessageType()==GetSensorsNow)
	{
		pthread_mutex_lock(simulations[simID]->queueLock);

		//put GetSensorsNow to queue
		simulations[simID]->commandQueue.push_back(*cmd);
		pthread_mutex_unlock(simulations[simID]->emptyQueueLock);

		pthread_mutex_unlock(simulations[simID]->queueLock);

		//wait until last sensor value is read by sensorReader-thread. reader thread unlocks this when ready (alternative: contition_wait)
		pthread_mutex_lock(simulations[simID]->sensorsDemanded);

		ret.setMessageACKGetSensors(simulations[simID]->gotSensorsNow->getDataStream(), simulations[simID]->gotSensorsNow->getByteSize());
	}

	pthread_mutex_unlock(man_lock);
	return ret;
}


unsigned int SimulationManager::waitForSimulationToFinish(unsigned int simID, unsigned int max_useconds)
{
	cout<<"MAN: Blocking caller until Simulation "<<simID<<" has finished (maximum time: "<<(max_useconds/1000000.0)<<" seconds)."<<endl;

	unsigned int useconds=0;
	bool replaced=false;

	//Attention: "Joining with a thread that has previously been joined results in undefined behavior."
	//pthread_join(simulations[simID]->sensorThread, NULL);
	while(pthread_mutex_trylock(simulations[simID]->deleteMeLock) != 0)	//solange sim nicht auf serverseite beendet wurde, also alle daten angekommen sind...
	{
		pthread_mutex_lock(simulations[simID]->pidLock);
		pid_t thisPID=simulations[simID]->PID;
		pthread_mutex_unlock(simulations[simID]->pidLock);

		if(thisPID > 0)
		{
			char cmdline[512];
			snprintf(cmdline, 512, "/proc/%d/cmdline", thisPID);
			ifstream in(cmdline);
			in.getline(cmdline, 512);
			in.close();

			if(strlen(cmdline) && strcmp(cmdline, simulations[simID]->cmdline)!=0)	//wenn PID korrekten kommandonamen hat, d.h. nicht überschrieben oder beendet wurde wurde, dann kille Sim nach Zeitlimit.
			{
				//Wenn Simulation über Zeitlimit ist, killen
				if(useconds>=max_useconds)
				{
					cout<<"MAN: Warning: Simulation "<<simID<<" exceeded timelimit. Killing simulation with PID "<<thisPID<<".\n";
					kill(thisPID, 9);

					break;
				}
			}
			else	//sim nicht im system vorhanden
			{
				cout<<"MAN: Simulation "<<simID<<" does not seem to run under this PID: "<<thisPID<<".\n";

				//break; // nicht breaken, da StopSimulation noch eintreffen kann! Sim kann sich selbst regulär beendet haben!

				//Wenn Simulation über Zeitlimit ist, ignorieren
				if(useconds>=max_useconds)
				{
					cout<<"MAN: Warning: Simulation "<<simID<<" exceeded timelimit. It did not report StopSimulation.\n";

					break;
				}
			}
		}
		else if(max_useconds != 0 && useconds>=max_useconds)
		{
			cout<<"MAN: Warning: Simulation "<<simID<<" exceeded timelimit but did not report any PID.\n";

			break;
		}

		pthread_mutex_unlock(man_lock);
		usleep(100000);
		useconds += 100000;
		pthread_mutex_lock(man_lock);

		if(simulations[simID]==NULL)
		{
			cout<<"MAN: Simulation "<<simID<<" has been deleted inbetween.\n";

			break;
		}

		//absichern, dass simulation nicht ersetzt wurde, während geschlafen wurde, indem StopSimulation eingetroffen und von einem anderen Thread StartSimulation aufgerufen wurde!
		if(!(simulations[simID]->shutDown))
		{
			cout<<"MAN: Simulation "<<simID<<" has been replaced inbetween.\n";

			replaced=true;
			break;
		}
	}

	if(simulations[simID]!=NULL && !replaced)
	{
		deleteSimulation(simID);
	}

	cout<<"MAN: Deleting simulation "<<simID<<" from the list of controlled simulations.\n";

	if(useconds > max_useconds)
		return 0;
	else
		return (max_useconds - useconds);
}

SimulationManager::SimulationManager() : stats(this)
{
	man_lock = new pthread_mutex_t();
	pthread_mutex_init(man_lock, NULL);

	pthread_mutex_lock(man_lock);

	activeSimulations=0;

	pthread_mutex_unlock(man_lock);

	loadPlugins();
}


SimulationManager::~SimulationManager()
{
	pthread_mutex_lock(man_lock);

	for(unsigned int i=0;i<simulations.size();i++)
	{
		if(validID(i))
		{
			waitForSimulationToFinish(i,0);
			deleteSimulation(i);
		}
	}

	pthread_mutex_unlock(man_lock);
	delete man_lock;
	exit(0);
}


// get software version as String
const char* SimulationManager::Statistics::version()
{
	return "Simulation Manager v0.1";
}

// get startup date/time of the server as String
const char* SimulationManager::Statistics::onlineSince()
{
	return "9:00, 01.01.2012";
}

// get total number of simulation slots
int SimulationManager::Statistics::numSlots()
{
	pthread_mutex_lock(man->man_lock);
	int ret = MAX_SIMS;
	pthread_mutex_unlock(man->man_lock);
	return ret;
}

// get number of currently busy slots
int SimulationManager::Statistics::numBusySlots()
{
	pthread_mutex_lock(man->man_lock);
	int ret=man->activeSimulations;
	pthread_mutex_unlock(man->man_lock);
	return ret;
}

// get amount of currently free memory (in MB)
double SimulationManager::Statistics::getMemFreeMB()
{
	rusage usage;
	getrusage(RUSAGE_SELF,&usage);

		//TODO: Gesamten Speicher auslesen
	return 24*1024-usage.ru_idrss/1024;
}

// get current CPU load
double SimulationManager::Statistics::currentLoad()
{
	double avg[3];
	getloadavg(avg,3);

	return avg[0];	//last 1 min
}

// get total average of CPU load (averaged over all times since startup at
// which the specified number of simulations were active simultaneously)
double SimulationManager::Statistics::averageLoad(int numBusySlots)
{
	double avg[3];
	getloadavg(avg,3);

	return avg[2];	//last 15 min
}

// total amount of non-idle cpu time in seconds used on all cores since startup
double SimulationManager::Statistics::totalCpuTimeSeconds()
{
	return 10032;
}

// total number rejected slot requests
int SimulationManager::Statistics::numSlotRequestsRejected()
{
	return 13;
}

// total number of simulations done (i.e. not including the currently running)
int SimulationManager::Statistics::numFinishedSimulations()
{
	return 10;
}

// total number of simulation steps so far in all simulations (done & active)
int SimulationManager::Statistics::numStepsInFinishedSimulations() {
	return 172832;
}

// name of simulation in the slot specified, null if the slot is not busy
const char* SimulationManager::Statistics::simulationName(int slotNumber)
{
	if (slotNumber == 0)
		return "MySimulation #1";
	if (slotNumber == 1)
		return "MySimulation #2";
	return NULL;
}

// number of simulation steps done so far in the simulation in the slot specified
int SimulationManager::Statistics::numStepsSoFar(int slotNumber)
{
	pthread_mutex_lock(man->man_lock);
	int ret;
	if(man->validID(slotNumber))
		ret= man->simulations[slotNumber]->getSteps();
	else ret=-1;
	pthread_mutex_unlock(man->man_lock);
	return ret;
}

// the average time in ms for processing a time step in the simulation in the slot specified
double SimulationManager::Statistics::averageStepTimeMiliSeconds(int slotNumber)
{
	if (slotNumber == 0)
		return 4.6;
	if (slotNumber == 1)
		return 5.5;
	return 0;
}


// get total cumulative time since startup for
// which the specified number of simulations were active simultaneously)
double SimulationManager::Statistics::activeNumSlotsTimeSeconds(int numBusySlots)
{
	if (numBusySlots == 0)
		return 12;
	if (numBusySlots == 1)
		return 3;
	if (numBusySlots == 2)
		return 45;
	if (numBusySlots == 3)
		return 8;
	return 0;
}

