/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.mycosem;

import org.openmuc.jsml.structures.Integer16;
import org.openmuc.jsml.structures.OctetString;

import de.ekut.informatik.kn.josef.helper.Helper;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemValue;

/**
 * Representation of the COSEM IC DATA [classID=1, classVersion=0]. Attribute 1:
 * logicalName. Attribute 2: value. No Method.
 * 
 * @author Daniel Fuchs
 */
public class Data extends InterfaceClass {

    /**
     * Value at index 2.
     */
    protected SML_CosemValue value;

    /**
     * Constructor.
     * 
     * @param logicalName
     */
    public Data(String logicalName) {
        this.logicalName = new OctetString(logicalName);
        this.classID = new Integer16((short) 1);
        this.classVersion = new Integer16((short) 0);
    }

    /**
     * Constructor.
     * 
     * @param logicalName
     * @param value
     */
    public Data(String logicalName, SML_CosemValue value) {
        this.logicalName = new OctetString(logicalName);
        this.classID = new Integer16((short) 1);
        this.classVersion = new Integer16((short) 0);
        this.value = value;
    }

    /**
     * Gets the value at index 2.
     * 
     * @return
     */
    public SML_CosemValue getValue() {
        return value;
    }

    /**
     * Sets the value at index 2.
     * 
     * @param value
     */
    public void setValue(SML_CosemValue value) {
        this.value = value;
    }

    @Override
    public SML_CosemValue getValue(int index) {
        switch (index) {
        case 1:
            return new SML_CosemValue(this.logicalName);
        case 2:
            return this.value;
        default:
            return null;
        }
    }

    @Override
    public boolean setValue(int index, SML_CosemValue value) {
        switch (index) {
        case 1: {
            if (value.getChoice() instanceof OctetString) {
                this.logicalName = (OctetString) value.getChoice();
                return true;
            } else {
                return false;
            }
        }
        case 2: {
            this.value = value;
            return true;
        }
        default:
            return false;
        }
    }

    @Override
    public boolean runMethod(int index) {
        // DATA has no method
        return false;
    }

    @Override
    public int countAttributes() {
        return 2;
    }

    @Override
    public int countMethods() {
        return 0;
    }

    @Override
    public String printPretty() {
        String s = "";
        s = s + "Object:DATA(classID:" + this.classID.getVal()
                + ",classVersion:" + this.classVersion.getVal() + ")";
        s = s + "[logicalName:" + this.logicalName + "]";
        s = s + "[value:" + Helper.getRawValue(this.value) + "]";
        return s;
    }

    @Override
    public String[] getAttributesAsString() {
        return new String[] { this.logicalName.toString(),
                this.value.toString() };
    }

}
