/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.mycosem;

import org.openmuc.jsml.structures.Integer16;
import org.openmuc.jsml.structures.OctetString;

import de.ekut.informatik.kn.josef.myjsml.SML_CosemValue;

/**
 * Abstract representation of a COSEM Interface Class (IC). Each IC has a
 * classID and a classVersion. Each instance of an IC has its logicalName at
 * index 1.
 * 
 * @author Daniel Fuchs
 * @author Michael Hoefling
 */
public abstract class InterfaceClass {

    /**
     * classID of the IC.
     */
    protected Integer16 classID;

    /**
     * classVersion of the IC.
     */
    protected Integer16 classVersion;

    /**
     * Logical name of the object at index 1.
     */
    protected OctetString logicalName;

    /**
     * Gets the classID.
     * 
     * @return
     */
    public Integer16 getClassID() {
        return classID;
    }

    /**
     * Sets the classID.
     * 
     * @param classID
     */
    public void setClassID(Integer16 classID) {
        this.classID = classID;
    }

    /**
     * Gets the classVersion.
     * 
     * @return
     */
    public Integer16 getClassVersion() {
        return classVersion;
    }

    /**
     * Sets the classVersion.
     * 
     * @param classVersion
     */
    public void setClassVersion(Integer16 classVersion) {
        this.classVersion = classVersion;
    }

    /**
     * Gets the logical name.
     * 
     * @return
     */
    public OctetString getLogicalName() {
        return logicalName;
    }

    /**
     * Sets the logical name.
     * 
     * @param logicalName
     */
    public void setLogicalName(OctetString logicalName) {
        this.logicalName = logicalName;
    }

    /**
     * Gets the value at specified index. Used for SML communication.
     * 
     * @param index
     * @return
     */
    public abstract SML_CosemValue getValue(int index);

    /**
     * Sets the value at specified index. Used for SML communication.
     * 
     * @param index
     * @param value
     */
    public abstract boolean setValue(int index, SML_CosemValue value);

    /**
     * Runs the method at specified index. Returns true if the index matched a
     * method, else false.
     * 
     * @param index
     * @return
     */
    public abstract boolean runMethod(int index);

    /**
     * Returns the number of attributes.
     * 
     * @return
     */
    public abstract int countAttributes();

    /**
     * Returns the number of methods.
     * 
     * @return
     */
    public abstract int countMethods();

    /**
     * Returns a String with a nice representation of the object.
     * 
     * @return
     */
    public abstract String printPretty();

    /**
     * Returns a simple String with the logical name and the class id. Used only
     * for the GUI.
     * 
     * @return
     */
    @Override
    public String toString() {
        return this.logicalName + "-" + this.classID.getVal();
    }

    /**
     * Checks if other IC equals this. Interface Classes are equal, if logical
     * name and class id are equal.
     * 
     * @param ic
     * @return
     */
    public boolean isEqual(InterfaceClass ic) {
        return (this.logicalName.toString().equals(ic.logicalName.toString())
                && this.classID.getVal() == ic.getClassID().getVal());
    }

    /**
     * Gets the attributes as a string representation. Used only for the GUI.
     */
    public abstract String[] getAttributesAsString();

    @Override
    public boolean equals(Object arg0) {
        if (arg0 == null) {
            return false;
        }
        if (arg0 == this) {
            return true;
        }
        if (!(arg0 instanceof InterfaceClass)) {
            return false;
        }

        InterfaceClass ic = (InterfaceClass) arg0;
        return isEqual(ic);
    }

}
