/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.mycosem;

import org.openmuc.jsml.structures.Integer16;
import org.openmuc.jsml.structures.OctetString;

import de.ekut.informatik.kn.josef.myjsml.SML_CosemValue;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemValueList;

/**
 * Representation of an object list element, as used in the COSEM Association LN
 * object. For simplicity, there is no field "access_rights".
 * 
 * @author Daniel Fuchs
 */
public class ObjectListElement {

    protected Integer16 classID;
    protected Integer16 classVersion;
    protected OctetString logicalName;

    /**
     * Constructor.
     */
    public ObjectListElement() {
    }

    /**
     * Constructor.
     * 
     * @param classID
     * @param classVersion
     * @param logicalName
     */
    public ObjectListElement(Integer16 classID, Integer16 classVersion,
            OctetString logicalName) {
        this.classID = classID;
        this.classVersion = classVersion;
        this.logicalName = logicalName;
    }

    public Integer16 getClassID() {
        return classID;
    }

    public void setClassID(Integer16 classID) {
        this.classID = classID;
    }

    public Integer16 getClassVersion() {
        return classVersion;
    }

    public void setClassVersion(Integer16 classVersion) {
        this.classVersion = classVersion;
    }

    public OctetString getLogicalName() {
        return logicalName;
    }

    public void setLogicalName(OctetString logicalName) {
        this.logicalName = logicalName;
    }

    /**
     * Deconstructs a SML_CosemValue, containing a representation of an object
     * list element. Sets the values in this object list element.
     * 
     * @param value
     */
    public boolean deconstructSML(SML_CosemValue value) {
        if (!(value.getChoice() instanceof SML_CosemValueList))
            return false;
        SML_CosemValueList outerList = (SML_CosemValueList) value.getChoice();
        SML_CosemValue[] array = outerList.getArray();
        // System.out.println("deconstructSML array length: "+array.length); //
        // debugging
        if (array.length != 3)
            return false;
        boolean success = true;
        if (array[0].getChoice() instanceof Integer16) {
            this.classID = (Integer16) array[0].getChoice();
        } else {
            success = false;
        }
        if (array[1].getChoice() instanceof Integer16) {
            this.classVersion = (Integer16) array[1].getChoice();
        } else {
            success = false;
        }
        if (array[2].getChoice() instanceof OctetString) {
            this.logicalName = (OctetString) array[2].getChoice();
        } else {
            success = false;
        }
        return success;
    }

    /**
     * Constructs a SML_CosemValue representation of the values in this object
     * list element.
     * 
     * @return
     */
    public SML_CosemValue constructSML() {
        SML_CosemValue id = new SML_CosemValue(this.classID);
        SML_CosemValue version = new SML_CosemValue(this.classVersion);
        SML_CosemValue name = new SML_CosemValue(this.logicalName);

        SML_CosemValue[] listElementInner = new SML_CosemValue[] { id, version,
                name };
        SML_CosemValueList listElementOuter = new SML_CosemValueList(
                listElementInner);
        return new SML_CosemValue(listElementOuter);
    }

    /**
     * Returns a nice String with information about this object list element.
     * 
     * @return
     */
    public String printPretty() {
        String s = "";
        s = s + "ObjectListElement[ClassID:" + this.classID.getVal()
                + ", ClassVersion:" + this.classVersion.getVal()
                + ", LogicalName:" + this.logicalName.toString() + "]";
        return s;
    }

}
