/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.mycosem;

import org.openmuc.jsml.structures.Integer16;
import org.openmuc.jsml.structures.Integer32;
import org.openmuc.jsml.structures.OctetString;

import de.ekut.informatik.kn.josef.helper.Helper;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemValue;

/**
 * Representation of the COSEM IC REGISTER [classID=3, classVersion=0].
 * Attribute 1: logicalName. Attribute 2: value. Attribute 2: scalerUnit. Method
 * 1: reset().
 * 
 * @author Daniel Fuchs
 */
public class Register extends InterfaceClass {

    /**
     * Value at index 2.
     */
    protected SML_CosemValue value;

    /**
     * Scaler unit at index 3. Default if not used: scaler=0, unit=255.
     */
    protected ScalerUnit scalerUnit = new ScalerUnit();

    /**
     * Default value for the value at index 2. Used for resetting. Can be set
     * for each instance inidividually. Default is 0 (Integer32).
     */
    protected SML_CosemValue defaultValue = new SML_CosemValue(
            new Integer32(0));

    /**
     * Constructor.
     * 
     * @param logicalName
     */
    public Register(String logicalName) {
        this.logicalName = new OctetString(logicalName);
        this.classID = new Integer16((short) 3);
        this.classVersion = new Integer16((short) 0);
    }

    /**
     * Constructor.
     * 
     * @param logicalName
     * @param value
     */
    public Register(String logicalName, SML_CosemValue value) {
        this.logicalName = new OctetString(logicalName);
        this.classID = new Integer16((short) 3);
        this.classVersion = new Integer16((short) 0);
        this.value = value;
    }

    public SML_CosemValue getValue() {
        return value;
    }

    public void setValue(SML_CosemValue value) {
        this.value = value;
    }

    public SML_CosemValue getDefaultValue() {
        return defaultValue;
    }

    public void setDefaultValue(SML_CosemValue defaultValue) {
        this.defaultValue = defaultValue;
    }

    public ScalerUnit getScalerUnit() {
        return scalerUnit;
    }

    public void setScalerUnit(ScalerUnit scalerUnit) {
        this.scalerUnit = scalerUnit;
    }

    /**
     * Method at index 1. Resets the value at index 1 to an instance specific
     * value.
     */
    public void reset() {
        this.value = this.defaultValue;
    }

    @Override
    public SML_CosemValue getValue(int index) {
        switch (index) {
        case 1:
            return new SML_CosemValue(this.logicalName);
        case 2:
            return this.value;
        case 3:
            return this.scalerUnit.constructSML();
        default:
            return null;
        }
    }

    @Override
    public boolean setValue(int index, SML_CosemValue value) {
        switch (index) {
        case 1: {
            if (value.getChoice() instanceof OctetString) {
                this.logicalName = (OctetString) value.getChoice();
                return true;
            } else {
                return false;
            }
        }
        case 2: {
            this.value = value;
            return true;
        }
        case 3:
            return this.scalerUnit.deconstructSML(value);
        default:
            return false;
        }
    }

    @Override
    public boolean runMethod(int index) {
        if (index == 1) {
            reset();
            return true;
        } else {
            return false;
        }
    }

    @Override
    public int countAttributes() {
        return 3;
    }

    @Override
    public int countMethods() {
        return 1;
    }

    @Override
    public String printPretty() {
        String s = "";
        s = s + "Object:REGISTER(classID:" + this.classID.getVal()
                + ",classVersion:" + this.classVersion.getVal() + ")";
        s = s + "[logicalName:" + this.logicalName + "]";
        s = s + "[value:" + Helper.getRawValue(this.value) + ", scalerUnit:"
                + this.scalerUnit.printPretty() + "]";
        return s;
    }

    @Override
    public String[] getAttributesAsString() {
        return new String[] { this.logicalName.toString(),
                this.value.toString(), this.scalerUnit.printPretty() };
    }

}
