/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.myjsml;

import java.io.DataInputStream;
import java.io.IOException;

import org.openmuc.jsml.structures.ASNObject;
import org.openmuc.jsml.structures.SML_AttentionRes;
import org.openmuc.jsml.structures.SML_GetListReq;
import org.openmuc.jsml.structures.SML_GetListRes;
import org.openmuc.jsml.structures.SML_GetProcParameterReq;
import org.openmuc.jsml.structures.SML_GetProcParameterRes;
import org.openmuc.jsml.structures.SML_GetProfileListReq;
import org.openmuc.jsml.structures.SML_GetProfileListRes;
import org.openmuc.jsml.structures.SML_GetProfilePackReq;
import org.openmuc.jsml.structures.SML_GetProfilePackRes;
import org.openmuc.jsml.structures.SML_MessageBody;
import org.openmuc.jsml.structures.SML_PublicCloseReq;
import org.openmuc.jsml.structures.SML_PublicCloseRes;
import org.openmuc.jsml.structures.SML_PublicOpenReq;
import org.openmuc.jsml.structures.SML_PublicOpenRes;
import org.openmuc.jsml.structures.SML_SetProcParameterReq;
import org.openmuc.jsml.structures.Unsigned32;

/**
 * This class extends the SML_MessageBody class with COSEM-service support
 * according to the SML standard version 1.04.
 * 
 * @author Daniel Fuchs
 * @author Michael Hoefling
 */
public class SML_COSEM_MessageBody extends SML_MessageBody {

    /**
     * Constants for GET-COSEM service handling
     */
    public static final int GetCosemRequest = 0x00000800,
            GetCosemResponse = 0x00000801;

    /**
     * Constants for SET-COSEM service handling
     */
    public static final int SetCosemRequest = 0x00000900,
            SetCosemResponse = 0x00000901;

    /**
     * Constants for ACTION-COSEM service handling
     */
    public static final int ActionCosemRequest = 0x00000A00,
            ActionCosemResponse = 0x00000A01;

    public SML_COSEM_MessageBody(int tag, ASNObject choice) {
        if (!(tag == OpenRequest || tag == OpenResponse || tag == CloseRequest
                || tag == CloseResponse || tag == AttentionResponse
                || tag == GetProfilePackRequest || tag == GetProfilePackResponse
                || tag == GetProfileListRequest || tag == GetProfileListResponse
                || tag == GetProcParameterRequest
                || tag == GetProcParameterResponse
                || tag == SetProcParameterRequest
                || tag == SetProcParameterResponse || tag == GetListRequest
                || tag == GetListResponse

        // Added by D.F. to support COSEM
                || tag == GetCosemRequest || tag == GetCosemResponse
                || tag == SetCosemRequest || tag == SetCosemResponse
                || tag == ActionCosemRequest || tag == ActionCosemResponse)) {

            throw new IllegalArgumentException(
                    "SML_COSEM_MessagBody: Wrong value for tag! " + tag
                            + " is not allowed.");
        }

        this.choice = choice;
        this.tag = new Unsigned32(tag);
        isSelected = true;
    }

    public SML_COSEM_MessageBody() {
        super();
    }

    @Override
    public boolean decode(DataInputStream is) throws IOException {
        byte tlField = is.readByte();
        if (isOptional && (tlField == 0x01)) {
            isSelected = false;
            return true;
        }
        if ((tlField & 0xff) != 0x72 || !tag.decode(is)) {
            return false;
        }

        switch (tag.getVal()) {
        case OpenRequest:
            choice = new SML_PublicOpenReq();
            break;
        case OpenResponse:
            choice = new SML_PublicOpenRes();
            break;
        case CloseRequest:
            choice = new SML_PublicCloseReq();
            break;
        case CloseResponse:
            choice = new SML_PublicCloseRes();
            break;
        case GetProfileListRequest:
            choice = new SML_GetProfileListReq();
            break;
        case GetProfileListResponse:
            choice = new SML_GetProfileListRes();
            break;
        case GetProfilePackRequest:
            choice = new SML_GetProfilePackReq();
            break;
        case GetProfilePackResponse:
            choice = new SML_GetProfilePackRes();
            break;
        case GetProcParameterRequest:
            choice = new SML_GetProcParameterReq();
            break;
        case GetProcParameterResponse:
            choice = new SML_GetProcParameterRes();
            break;
        case SetProcParameterRequest:
            choice = new SML_SetProcParameterReq();
            break;
        case GetListRequest:
            choice = new SML_GetListReq();
            break;
        case GetListResponse:
            choice = new SML_GetListRes();
            break;
        case AttentionResponse:
            choice = new SML_AttentionRes();
            break;
        case GetCosemRequest:
            choice = new SML_GetCosemReq();
            break;
        case GetCosemResponse:
            choice = new SML_GetCosemRes();
            break;
        case SetCosemRequest:
            choice = new SML_SetCosemReq();
            break;
        case SetCosemResponse:
            choice = new SML_SetCosemRes();
            break;
        case ActionCosemRequest:
            choice = new SML_ActionCosemReq();
            break;
        case ActionCosemResponse:
            choice = new SML_ActionCosemRes();
            break;
        default:
            return false;
        }

        if (!choice.decode(is)) {
            return false;
        }

        isSelected = true;

        return true;
    }

}
