/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.smartmeter.cosem;

import org.openmuc.jsml.structures.Integer16;
import org.openmuc.jsml.structures.OctetString;
import org.openmuc.jsml.structures.SML_AttentionRes;
import org.openmuc.jsml.structures.SML_Message;
import org.openmuc.jsml.structures.SML_PublicCloseRes;
import org.openmuc.jsml.structures.SML_PublicOpenReq;
import org.openmuc.jsml.structures.SML_PublicOpenRes;
import org.openmuc.jsml.structures.Unsigned8;

import de.ekut.informatik.kn.josef.helper.Helper;
import de.ekut.informatik.kn.josef.mycosem.InterfaceClass;
import de.ekut.informatik.kn.josef.mycosem.LogicalDevice;
import de.ekut.informatik.kn.josef.mycosem.ServerModel;
import de.ekut.informatik.kn.josef.myjsml.SML_COSEM_MessageBody;
import de.ekut.informatik.kn.josef.myjsml.SML_ActionCosemReq;
import de.ekut.informatik.kn.josef.myjsml.SML_ActionCosemRes;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemAttrList;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemAttribute;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemAttributeContent;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemAttributeDesc;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemValue;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemValueList;
import de.ekut.informatik.kn.josef.myjsml.SML_GetCosemReq;
import de.ekut.informatik.kn.josef.myjsml.SML_GetCosemRes;
import de.ekut.informatik.kn.josef.myjsml.SML_SetCosemReq;
import de.ekut.informatik.kn.josef.myjsml.SML_SetCosemRes;

/**
 * Handles SML request messages. Executes services on the COSEM server model.
 * Responses the SML requests with SML responses.
 * 
 * @author Daniel Fuchs
 */
public class RequestHandler {

    /**
     * Use autentication?
     */
    private boolean useAuthentication;

    /**
     * Username for authentication.
     */
    private OctetString authUsername;

    /**
     * Password for authentication.
     */
    private OctetString authPassword;

    /**
     * Holds a reference to the server model.
     */
    private ServerModel serverModel;

    /**
     * Constructor. Use without authentication.
     */
    public RequestHandler() {
        this.useAuthentication = false;
    }

    /**
     * Constructor. Use with authentication.
     * 
     * @param clientUsername
     * @param clientPassword
     */
    public RequestHandler(OctetString clientUsername,
            OctetString clientPassword) {
        this.useAuthentication = true;
        this.authUsername = clientUsername;
        this.authPassword = clientPassword;
    }

    /**
     * Sets the reference to the server model.
     * 
     * @param sm
     */
    public void setServerModel(ServerModel sm) {
        this.serverModel = sm;
    }

    /**
     * Handles SML OpenRequest messages.
     * 
     * @param message
     *            OpenRequest
     * @return OpenResponse
     */
    public SML_Message handleOpenRequest(SML_Message message) {
        if (SML_COSEM_MessageBody.OpenRequest != message.getMessageBody()
                .getTag().getVal()) {
            System.out.println(
                    "Message put in handler was not an OpenRequest, return null.");
            return null;
        }
        SML_PublicOpenReq openRequest = (SML_PublicOpenReq) message
                .getMessageBody().getChoice();

        // OpenResponse fields
        OctetString codepage = openRequest.getCodepage(); // OPTIONAL
        OctetString clientId = openRequest.getClientId(); // OPTIONAL
        OctetString reqFileId = openRequest.getReqFileId();
        OctetString serverId = openRequest.getServerId();
        Unsigned8 smlVersion = openRequest.getSmlVersion(); // OPTIONAL

        // SML message fields
        OctetString transactionID = message.getTransactionId();
        Unsigned8 groupNo = message.getGroupNo();
        Unsigned8 abortOnError = message.getAbortOnError();

        // check authentication
        OctetString username = openRequest.getUsername();
        OctetString password = openRequest.getPassword();
        if (!checkAuthentication(username, password)) {
            return createClientNotAuthResponse(serverId, transactionID, groupNo,
                    abortOnError);
        }

        SML_PublicOpenRes openResponse = new SML_PublicOpenRes(codepage,
                clientId, reqFileId, serverId, null, smlVersion);
        SML_COSEM_MessageBody openBody = new SML_COSEM_MessageBody(
                SML_COSEM_MessageBody.OpenResponse, openResponse);
        SML_Message openResponseMessage = new SML_Message(transactionID,
                groupNo, abortOnError, openBody);
        return openResponseMessage;
    }

    /**
     * Handles SML CloseRequest messages.
     * 
     * @param message
     *            CloseRequest
     * @return CloseResponse
     */
    public SML_Message handleCloseRequest(SML_Message message) {
        if (SML_COSEM_MessageBody.CloseRequest != message.getMessageBody()
                .getTag().getVal()) {
            System.out.println(
                    "Message put in handler was not an CloseRequest, return null.");
            return null;
        }
        // optional, no further use
        // SML_PublicCloseReq closeRequest = (SML_PublicCloseReq)
        // message.getMessageBody().getChoice();
        // SML_Signature globalSignature = (SML_Signature)
        // closeRequest.getGlobalSignature();

        // SML message fields
        OctetString transactionID = message.getTransactionId();
        Unsigned8 groupNo = message.getGroupNo();
        Unsigned8 abortOnError = message.getAbortOnError();

        SML_PublicCloseRes closeResponse = new SML_PublicCloseRes(null);
        SML_COSEM_MessageBody closeBody = new SML_COSEM_MessageBody(
                SML_COSEM_MessageBody.CloseResponse, closeResponse);
        SML_Message closeResponseMessage = new SML_Message(transactionID,
                groupNo, abortOnError, closeBody);
        return closeResponseMessage;
    }

    /**
     * Handles SML GetCosemRequest messages.
     * 
     * @param message
     *            GetCosemRequest
     * @param objList
     * @return GetCosemResponse or AttentionResponse
     */
    public SML_Message handleGetCosemRequest(SML_Message message) {
        if (SML_COSEM_MessageBody.GetCosemRequest != message.getMessageBody()
                .getTag().getVal()) {
            System.out.println(
                    "Message put in handler was not an GetCosemRequest, return null.");
            return null;
        }
        SML_GetCosemReq getRequest = (SML_GetCosemReq) message.getMessageBody()
                .getChoice();

        // GetCosemResponse fields
        OctetString clientId = getRequest.getClientId(); // OPTIONAL
        OctetString serverId = getRequest.getServerId();
        OctetString objName = getRequest.getObjName();
        Integer16 classID = getRequest.getClassID();
        Integer16 classVersion = getRequest.getClassVersion();

        // SML message fields
        OctetString transactionID = message.getTransactionId();
        Unsigned8 groupNo = message.getGroupNo();
        Unsigned8 abortOnError = message.getAbortOnError();

        // check authentication
        OctetString username = getRequest.getUsername();
        OctetString password = getRequest.getPassword();
        if (!checkAuthentication(username, password)) {
            return createClientNotAuthResponse(serverId, transactionID, groupNo,
                    abortOnError);
        }

        InterfaceClass obj = findObjectInLdev(classID, objName, serverId); // retrieve
                                                                           // object
                                                                           // from
                                                                           // logical
                                                                           // device
        if (obj == null) {
            return createNotFoundResponse(serverId, transactionID, groupNo,
                    abortOnError);
        }

        // list of attributes for the response
        SML_CosemAttribute[] responseAttributes = null;
        if (getRequest.getAttributeIndexList().isSelected()) { // if only some
                                                               // attributes are
                                                               // requested,
                                                               // retrieve
                                                               // attribute
                                                               // index
            SML_CosemAttributeDesc[] attributeDescriptions = getRequest
                    .getAttributeIndexList().getAttributeDescription(); // requested
                                                                        // attribute
                                                                        // indexes
                                                                        // in
                                                                        // array
            responseAttributes = new SML_CosemAttribute[attributeDescriptions.length];
            for (int i = 0; i < attributeDescriptions.length; i++) { // for each
                                                                     // requested
                                                                     // attribute
                                                                     // index
                short reqAttributeIndex = attributeDescriptions[i]
                        .getAttributeIndex().getVal(); // index of a requested
                                                       // attribute
                SML_CosemValue data = obj.getValue(reqAttributeIndex); // retrieve
                                                                       // value
                if (data == null) { // if no att with given index, return
                                    // attention response
                    System.out.println(
                            "handleGetCosemRequest: selective attribute at index "
                                    + reqAttributeIndex
                                    + " was not retrieved!");
                    return createNotFoundResponse(serverId, transactionID,
                            groupNo, abortOnError);
                } else {
                    System.out.println(
                            "handleGetCosemRequest: selective attribute at index "
                                    + reqAttributeIndex + " was retrieved!");
                    SML_CosemAttributeContent attributeContent = new SML_CosemAttributeContent(
                            SML_CosemAttributeContent.DATA, data);
                    responseAttributes[i] = new SML_CosemAttribute(
                            new SML_CosemAttributeDesc(
                                    new Integer16(reqAttributeIndex), null),
                            attributeContent);
                }
            }
        } else { // retrieve all attributes
            responseAttributes = new SML_CosemAttribute[obj.countAttributes()];
            for (int i = 1; i <= obj.countAttributes(); i++) { // object index
                                                               // always starts
                                                               // at 1
                SML_CosemValue data = obj.getValue(i); // retrieve value
                if (data == null) { // if no att with given index, return
                                    // attention response
                    return createNotFoundResponse(serverId, transactionID,
                            groupNo, abortOnError);
                } else {
                    SML_CosemAttributeContent attributeContent = new SML_CosemAttributeContent(
                            SML_CosemAttributeContent.DATA, data);
                    SML_CosemAttributeDesc attributeDescription = new SML_CosemAttributeDesc(
                            new Integer16((short) i), null);
                    responseAttributes[i - 1] = new SML_CosemAttribute(
                            attributeDescription, attributeContent);
                    ; // -1 because array index starts at 0
                }
            }
        }
        SML_CosemAttrList attributeList = new SML_CosemAttrList(
                responseAttributes);
        SML_GetCosemRes getResponse = new SML_GetCosemRes(clientId, serverId,
                objName, classID, classVersion, attributeList);
        SML_COSEM_MessageBody getResponseBody = new SML_COSEM_MessageBody(
                SML_COSEM_MessageBody.GetCosemResponse, getResponse);
        SML_Message getResponseMessage = new SML_Message(transactionID, groupNo,
                abortOnError, getResponseBody);
        return getResponseMessage;
    }

    /**
     * Handles SML SetCosemRequest messages.
     * 
     * @param message
     *            SetCosemRequest
     * @param objList
     * @return SetCosemResponse or AttentionResponse
     */
    public SML_Message handleSetCosemRequest(SML_Message message) {
        if (SML_COSEM_MessageBody.SetCosemRequest != message.getMessageBody()
                .getTag().getVal()) {
            System.out.println(
                    "Message put in handler was not a SetCosemRequest, return null.");
            return null;
        }
        SML_SetCosemReq setRequest = (SML_SetCosemReq) message.getMessageBody()
                .getChoice();

        // SetCosemResponse fields
        OctetString clientId = setRequest.getClientId(); // OPTIONAL
        OctetString serverId = setRequest.getServerId();
        OctetString objName = setRequest.getObjName();
        Integer16 classID = setRequest.getClassID();
        Integer16 classVersion = setRequest.getClassVersion();
        // SML_CosemAttrList(responseAttributes); // OPTIONAL

        // SML message fields
        OctetString transactionID = message.getTransactionId();
        Unsigned8 groupNo = message.getGroupNo();
        Unsigned8 abortOnError = message.getAbortOnError();

        // check authentication
        OctetString username = setRequest.getUsername();
        OctetString password = setRequest.getPassword();
        if (!checkAuthentication(username, password)) {
            return createClientNotAuthResponse(serverId, transactionID, groupNo,
                    abortOnError);
        }

        InterfaceClass obj = findObjectInLdev(classID, objName, serverId); // retrieve
                                                                           // object
                                                                           // from
                                                                           // logical
                                                                           // device
        if (obj == null) {
            return createNotFoundResponse(serverId, transactionID, groupNo,
                    abortOnError);
        }

        SML_CosemAttribute[] attributeArray = setRequest.getAttributeList()
                .getCosemAttribute();
        for (int i = 0; i < attributeArray.length; i++) {
            short attributeIndex = attributeArray[i].getAttributeDescription()
                    .getAttributeIndex().getVal(); // index to set
            // if content is type DATA
            if (SML_CosemAttributeContent.DATA == attributeArray[i]
                    .getAttributeContent().getTag().getVal()) {
                if (attributeIndex > obj.countAttributes()
                        || attributeIndex == 0) { // if requested attribute
                                                  // index is out of range
                    System.out.println(
                            "handleSetCosemRequest: requested attribute index out of range.");
                    return createNotFoundResponse(serverId, transactionID,
                            groupNo, abortOnError);
                } else {
                    SML_CosemValue value = (SML_CosemValue) attributeArray[i]
                            .getAttributeContent().getChoice(); // value to set
                    if (obj.setValue(attributeIndex, value)) {
                        System.out.println(
                                "handleSetCosemRequest: new value at index: "
                                        + attributeIndex + " was set!");
                    } else {
                        System.out.println(
                                "handleSetCosemRequest: new value at index: "
                                        + attributeIndex + " was not set!");
                        return createNotWrittenResponse(serverId, transactionID,
                                groupNo, abortOnError);
                    }
                }
            } else {
                // content is type DATAACCESSRESULT, not supported
            }
        }
        SML_SetCosemRes setResponse = new SML_SetCosemRes(clientId, serverId,
                objName, classID, classVersion, null);
        SML_COSEM_MessageBody setResponseBody = new SML_COSEM_MessageBody(
                SML_COSEM_MessageBody.SetCosemResponse, setResponse);
        SML_Message setResponseMessage = new SML_Message(transactionID, groupNo,
                abortOnError, setResponseBody);
        return setResponseMessage;
    }

    /**
     * Handles SML ActionCosemRequest messages.
     * 
     * @param message
     *            ActionCosemRequest
     * @param objList
     * @return ActionCosemResponse or AttentionResponse
     */
    public SML_Message handleActionCosemRequest(SML_Message message) {
        if (SML_COSEM_MessageBody.ActionCosemRequest != message.getMessageBody()
                .getTag().getVal()) {
            System.out.println(
                    "Message put in handler was not an ActionCosemRequest, return null.");
            return null;
        }
        SML_ActionCosemReq actionRequest = (SML_ActionCosemReq) message
                .getMessageBody().getChoice();

        // ActionCosemResponse fields
        OctetString clientId = actionRequest.getClientId(); // OPTIONAL
        OctetString serverId = actionRequest.getServerId();
        OctetString objName = actionRequest.getObjName();
        Integer16 classID = actionRequest.getClassID();
        Integer16 classVersion = actionRequest.getClassVersion();
        // SML_CosemAttrList attributeList = new
        // SML_CosemAttrList(responseAttributes); // OPTIONAL

        // SML message fields
        OctetString transactionID = message.getTransactionId();
        Unsigned8 groupNo = message.getGroupNo();
        Unsigned8 abortOnError = message.getAbortOnError();

        // check authentication
        OctetString username = actionRequest.getUsername();
        OctetString password = actionRequest.getPassword();
        if (!checkAuthentication(username, password)) {
            return createClientNotAuthResponse(serverId, transactionID, groupNo,
                    abortOnError);
        }

        InterfaceClass obj = findObjectInLdev(classID, objName, serverId); // retrieve
                                                                           // object
                                                                           // from
                                                                           // logical
                                                                           // device
        if (obj == null) {
            return createNotFoundResponse(serverId, transactionID, groupNo,
                    abortOnError);
        }

        // handle action request
        boolean wasRun = false;
        int serviceIndex = actionRequest.getServiceIndex().getVal(); // index of
                                                                     // the
                                                                     // method
                                                                     // to run
        SML_CosemValue serviceParameter = actionRequest.getServiceParameter(); // parameter(s)
                                                                               // to
                                                                               // run
                                                                               // the
                                                                               // method
                                                                               // with
        if (serviceParameter.getChoice() instanceof SML_CosemValueList) { // if
                                                                          // there
                                                                          // is
                                                                          // a
                                                                          // list
                                                                          // of
                                                                          // parameters
            // SML_CosemValueList valueList = (SML_CosemValueList)
            // serviceParameter.getChoice();
            // functionality not supported, because there are currently no
            // methods with parameters.
            System.out.println(
                    "handleActionCosemRequest: serviceParameter(s) not supported.");
        } else { // if there is no parameter
            wasRun = obj.runMethod(serviceIndex); // put in serviceParameter
            if (wasRun) {
                System.out.println("handleActionCosemRequest: method at index "
                        + serviceIndex + " was run!");
            } else {
                System.out.println("handleActionCosemRequest: method at index "
                        + serviceIndex + " was not run!");
            }
        }

        if (wasRun) { // successfully run
            SML_ActionCosemRes actionResponse = new SML_ActionCosemRes(clientId,
                    serverId, objName, classID, classVersion, null);
            SML_COSEM_MessageBody actionResponseBody = new SML_COSEM_MessageBody(
                    SML_COSEM_MessageBody.ActionCosemResponse, actionResponse);
            SML_Message actionResponseMessage = new SML_Message(transactionID,
                    groupNo, abortOnError, actionResponseBody);
            return actionResponseMessage;
        } else { // method was not run
            OctetString msg = new OctetString(
                    "Method was not started, not available or invalid index.");
            SML_AttentionRes att = new SML_AttentionRes(serverId,
                    SML_AttentionRes.ONE_OR_MORE_DESTITNATION_ATTRIBUTES_CANNOT_BE_WRITTEN,
                    msg, null);
            SML_COSEM_MessageBody body = new SML_COSEM_MessageBody(
                    SML_COSEM_MessageBody.AttentionResponse, att);
            SML_Message responseMsg = new SML_Message(transactionID, groupNo,
                    abortOnError, body);
            return responseMsg;
        }
    }

    /**
     * Finds an object in a logical device.
     * 
     * @param classID
     * @param objName
     * @param serverID
     * @return InterfaceClass if object was found, else null
     */
    private InterfaceClass findObjectInLdev(Integer16 classID,
            OctetString objName, OctetString serverID) {
        LogicalDevice ldev = serverModel.getLdev(serverID);
        if (ldev != null) {
            return Helper.findObjectInList(classID, objName,
                    ldev.getObjectList());
        }
        return null;
    }

    /**
     * Check if authentication is true.
     * 
     * @param username
     * @param password
     * @return
     */
    private boolean checkAuthentication(OctetString username,
            OctetString password) {
        if (!useAuthentication)
            return true;
        return (username.equals(authUsername) && password.equals(authPassword));
    }

    /**
     * Creates an attention response for an unauthenticated client.
     * 
     * @param serverID
     * @param transactionID
     * @param groupNo
     * @param abortOnError
     * @return
     */
    private SML_Message createClientNotAuthResponse(OctetString serverID,
            OctetString transactionID, Unsigned8 groupNo,
            Unsigned8 abortOnError) {
        System.out
                .println("Client is not authentic, return AttentionResponse!");
        OctetString msg = new OctetString("Client was not authenticated.");
        SML_AttentionRes att = new SML_AttentionRes(serverID,
                SML_AttentionRes.INADEQUATE_AUTHENTICATION, msg, null);
        SML_COSEM_MessageBody body = new SML_COSEM_MessageBody(
                SML_COSEM_MessageBody.AttentionResponse, att);
        SML_Message responseMsg = new SML_Message(transactionID, groupNo,
                abortOnError, body);
        return responseMsg;
    }

    /**
     * Creates an attention response if no COSEM object, attribute or method was
     * found.
     * 
     * @param serverID
     * @param transactionID
     * @param groupNo
     * @param abortOnError
     * @return
     */
    private SML_Message createNotFoundResponse(OctetString serverID,
            OctetString transactionID, Unsigned8 groupNo,
            Unsigned8 abortOnError) {
        System.out.println(
                "No matching COSEM object, attribute or method was found!");
        OctetString msg = new OctetString(
                "No matching COSEM object, attribute or method was found.");
        SML_AttentionRes att = new SML_AttentionRes(serverID,
                SML_AttentionRes.ONE_OR_MORE_DESTITNATION_ATTRIBUTES_CANNOT_BE_READ,
                msg, null);
        SML_COSEM_MessageBody body = new SML_COSEM_MessageBody(
                SML_COSEM_MessageBody.AttentionResponse, att);
        SML_Message responseMsg = new SML_Message(transactionID, groupNo,
                abortOnError, body);
        return responseMsg;
    }

    /**
     * Creates an attention response if no COSEM attribute can be written.
     * 
     * @param serverID
     * @param transactionID
     * @param groupNo
     * @param abortOnError
     * @return
     */
    private SML_Message createNotWrittenResponse(OctetString serverID,
            OctetString transactionID, Unsigned8 groupNo,
            Unsigned8 abortOnError) {
        System.out.println("No COSEM attribute can be written!");
        OctetString msg = new OctetString("No COSEM attribute can be written!");
        SML_AttentionRes att = new SML_AttentionRes(serverID,
                SML_AttentionRes.ONE_OR_MORE_DESTITNATION_ATTRIBUTES_CANNOT_BE_WRITTEN,
                msg, null);
        SML_COSEM_MessageBody body = new SML_COSEM_MessageBody(
                SML_COSEM_MessageBody.AttentionResponse, att);
        SML_Message responseMsg = new SML_Message(transactionID, groupNo,
                abortOnError, body);
        return responseMsg;
    }

}
