/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.smartmeter.cosem;

import java.util.ArrayList;
import java.util.Date;

import org.openmuc.jsml.structures.Integer32;
import org.openmuc.jsml.structures.Integer64;
import org.openmuc.jsml.structures.OctetString;
import org.openmuc.jsml.structures.Unsigned64;

import de.ekut.informatik.kn.josef.mycosem.ServerModel;
import de.ekut.informatik.kn.josef.mycosem.AssociationLN;
import de.ekut.informatik.kn.josef.mycosem.Data;
import de.ekut.informatik.kn.josef.mycosem.InterfaceClass;
import de.ekut.informatik.kn.josef.mycosem.LogicalDevice;
import de.ekut.informatik.kn.josef.mycosem.Register;
import de.ekut.informatik.kn.josef.mycosem.ScalerUnit;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemValue;

/**
 * This class models a smart meter as a COSEM server. For simplification, only
 * one logical device (management) is used currently.
 * 
 * @author Daniel Fuchs
 * @author Michael Hoefling
 */
public class SmartMeterModel extends ServerModel {

    /**
     * Constructor. Default server model.
     */
    public SmartMeterModel() {
        super();
        initializeMLD();
    }

    /**
     * Initializes the management logical device (MLD). Models the MLD and the
     * contained objects. The objects are based on the Dutch Smart Metering
     * Readout (DSMR) Companion Standard.
     */
    private void initializeMLD() {
        ArrayList<InterfaceClass> objList = new ArrayList<InterfaceClass>();

        // create Association LN object
        AssociationLN associationLN = new AssociationLN("0.0.40.0.0.255");
        objList.add(associationLN);

        // create Logical Device Name object
        OctetString mldName = new OctetString("UTB" + new Date().getTime());
        Data logicalDeviceName = new Data("0.0.42.0.0.255",
                new SML_CosemValue(mldName));
        objList.add(logicalDeviceName);

        // create Unix Clock object
        Data unixClock = new Data("0.0.1.1.0.255",
                new SML_CosemValue(new Integer64(new Date().getTime())));
        objList.add(unixClock);

        // create Register object: "meter reading electricity delivered to the
        // household" (import +P)
        Register electricityDelivered = new Register("1.0.1.8.1.255",
                new SML_CosemValue(new Unsigned64(0)));
        electricityDelivered.setScalerUnit(
                new ScalerUnit(new Integer32(-6), new Integer32(30))); // unit
                                                                       // 30 =
                                                                       // active
                                                                       // energy
                                                                       // (Wh),
                                                                       // scales
                                                                       // to kWh
        electricityDelivered
                .setDefaultValue(new SML_CosemValue(new Unsigned64(0)));
        objList.add(electricityDelivered);

        // create Register object: "meter reading electricity generated by the
        // household" (export -P)
        Register electricityGenerated = new Register("1.0.2.8.1.255",
                new SML_CosemValue(new Unsigned64(0)));
        electricityGenerated.setScalerUnit(
                new ScalerUnit(new Integer32(-6), new Integer32(30)));
        electricityGenerated
                .setDefaultValue(new SML_CosemValue(new Unsigned64(0)));
        objList.add(electricityGenerated);

        // create Register object: "instanteneous electricity power delivered to
        // the household" (actual import +P at the moment)
        Register instantPowerDelivered = new Register("1.0.1.7.0.255",
                new SML_CosemValue(new Unsigned64(0)));
        instantPowerDelivered.setScalerUnit(
                new ScalerUnit(new Integer32(-3), new Integer32(27))); // unit
                                                                       // 27 =
                                                                       // active
                                                                       // power
                                                                       // (W),
                                                                       // scales
                                                                       // to W
        instantPowerDelivered
                .setDefaultValue(new SML_CosemValue(new Unsigned64(0)));
        objList.add(instantPowerDelivered);

        // create Register object: "instanteneous electricity power generated by
        // the household" (actual export -P at the moment)
        Register instantPowerGenerated = new Register("1.0.2.7.0.255",
                new SML_CosemValue(new Unsigned64(0)));
        instantPowerGenerated.setScalerUnit(
                new ScalerUnit(new Integer32(-3), new Integer32(27)));
        instantPowerGenerated
                .setDefaultValue(new SML_CosemValue(new Unsigned64(0)));
        objList.add(instantPowerGenerated);

        // create Register object: "instanteneous voltage in Volt"
        Register instantVoltage = new Register("1.0.12.7.0.255",
                new SML_CosemValue(new Integer32(0)));
        instantVoltage.setScalerUnit(
                new ScalerUnit(new Integer32(-2), new Integer32(35))); // unit
                                                                       // 35 =
                                                                       // Volt,
                                                                       // scales
                                                                       // to
                                                                       // Volt
        instantVoltage.setDefaultValue(new SML_CosemValue(new Integer32(0)));
        objList.add(instantVoltage);

        // finally, refresh the object list hold by the associationLN object.
        associationLN.refreshObjectList(objList); // refresh after each added or
                                                  // deleted object.

        // create the management logical device and add it to the model.
        LogicalDevice mld = new LogicalDevice(new OctetString("1"), mldName);
        mld.getObjectList().addAll(objList);
        getLdevs().add(mld);
    }

}
