/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.smartmeter.simulation;

import java.util.ArrayList;

import org.openmuc.jsml.structures.Integer16;
import org.openmuc.jsml.structures.Integer32;
import org.openmuc.jsml.structures.OctetString;
import org.openmuc.jsml.structures.Unsigned64;

import de.ekut.informatik.kn.josef.helper.Helper;
import de.ekut.informatik.kn.josef.mycosem.ServerModel;
import de.ekut.informatik.kn.josef.mycosem.InterfaceClass;
import de.ekut.informatik.kn.josef.myjsml.SML_CosemValue;

/**
 * This class implements a load curve generator for a smart meter. Based on
 * standard load profiles, pseudo-randomized load curves are generated and fed
 * to the COSEM model of the smart meter.
 * 
 * @author Daniel Fuchs
 * @author Michael Hoefling
 */
public class LoadCurveGenerator extends Thread {

    /**
     * Holds a reference to the server model.
     */
    private ServerModel serverModel;

    /**
     * Number of seconds in realtime, that a simulated quarter-hour has. 900 for
     * real-time = simulation-time. Default is 4.
     */
    private int secPerQuarterHour = 4;

    /**
     * Resolution of the simulation in milliseconds. Default is 1000.
     */
    private long simulationResolution = 1000;

    /**
     * H0 standard load profile of a household.
     */
    private ArrayList<Double> profileH0 = LoadProfileReader.readLoadProfile(0,
            "/de/ekut/informatik/kn/josef/smartmeter/loadprofiles/SLP_H0_Tag");

    /**
     * EV0 standard input profile of a photovoltaic station.
     */
    private ArrayList<Double> profileEV0 = LoadProfileReader.readLoadProfile(0,
            "/de/ekut/informatik/kn/josef/smartmeter/loadprofiles/SEP_EV0_Mai");

    /**
     * Current H0 profile index.
     */
    private int profileH0Index = 0;
    /**
     * Current EV0 profile index.
     */
    private int profileEV0Index = 0;

    /**
     * Annual electricity consumption in kWh, utilized to extrapolate the H0
     * profile. Default is 1000 kWh.
     */
    private double annualH0 = 1000.0;
    /**
     * Annual electricity production in kWh, utilized to extrapolate the EV0
     * profile Default is 1000 kWh.
     */
    private double annualEV0 = 1000.0;

    /**
     * Constructor.
     * 
     * @param serverModel
     * @param secondsPerQuarterHour
     */
    public LoadCurveGenerator(ServerModel serverModel,
            int secondsPerQuarterHour) {
        this.serverModel = serverModel;
        if (secondsPerQuarterHour > 0) {
            this.secPerQuarterHour = secondsPerQuarterHour;
        }
    }

    /**
     * Sets the simulated, annual consumption.
     * 
     * @param annualElectricityConsumption
     */
    public void setAnnualH0(double consumption) {
        this.annualH0 = consumption;
    }

    /**
     * Sets the simulated, annual production.
     * 
     * @param production
     */
    public void setAnnualEV0(double production) {
        this.annualEV0 = production;
    }

    /**
     * Gets the simulated, annual consumption.
     * 
     * @return
     */
    public double getAnnualH0() {
        return annualH0;
    }

    /**
     * Gets the simulated, annual production.
     * 
     * @return
     */
    public double getAnnualEV0() {
        return annualEV0;
    }

    /**
     * Gets the seconds per quarter hour.
     * 
     * @return
     */
    public int getSecPerQuarterHour() {
        return this.secPerQuarterHour;
    }

    /**
     * Gets the simulation resolution.
     * 
     * @return
     */
    public long getSimulationResolution() {
        return this.simulationResolution;
    }

    /**
     * Simulates the next step. Reads objects from the server model and writes
     * new values in predefined objects.
     * 
     * @param objList
     *            object list
     */
    private void simulateNextStep(ArrayList<InterfaceClass> objList) {
        // get the "unix clock" object and set the current time
        InterfaceClass c = Helper.findObjectInList(new Integer16((short) 1),
                new OctetString("0.0.1.1.0.255"), objList);
        if (c != null) {
            c.setValue(2, new SML_CosemValue(new Unsigned64(Helper.getTime())));
        }

        // simulate the load profile values
        int currentH0Index = simulateProfileIndex(this.secPerQuarterHour,
                this.profileH0);
        double h0Value = getProfileValue(currentH0Index, this.profileH0)
                * (this.annualH0 / 1000);
        int currentEV0Index = simulateProfileIndex(this.secPerQuarterHour,
                this.profileEV0);
        double ev0Value = getProfileValue(currentEV0Index, this.profileEV0)
                * (this.annualEV0 / 1000);

        // get the "meter reading electricity delivered to the household" object
        c = Helper.findObjectInList(new Integer16((short) 3),
                new OctetString("1.0.1.8.1.255"), objList);
        if (c != null) {
            if (currentH0Index == this.profileH0Index) {
                // do nothing
            } else {
                this.profileH0Index = currentH0Index;
                long val = Helper.getNumberValue(c.getValue(2)).longValue();
                val += h0Value;
                c.setValue(2, new SML_CosemValue(new Unsigned64(val)));
            }
        }

        // get the "meter reading electricity generated by the household" object
        c = Helper.findObjectInList(new Integer16((short) 3),
                new OctetString("1.0.2.8.1.255"), objList);
        if (c != null) {
            if (currentEV0Index == this.profileEV0Index) {
                // do nothing
            } else {
                this.profileEV0Index = currentEV0Index;
                long val = Helper.getNumberValue(c.getValue(2)).longValue();
                val += ev0Value;
                c.setValue(2, new SML_CosemValue(new Unsigned64(val)));
            }
        }

        // get the "actual instanteneous power import" object
        c = Helper.findObjectInList(new Integer16((short) 3),
                new OctetString("1.0.1.7.0.255"), objList);
        if (c != null) {
            long val = (long) LoadCurveGenerator.scatter(h0Value, 0.01); // scatter
            // the
            // value
            c.setValue(2, new SML_CosemValue(new Unsigned64(val)));
        }

        // get the "actual instanteneous power export" object
        c = Helper.findObjectInList(new Integer16((short) 3),
                new OctetString("1.0.2.7.0.255"), objList);
        if (c != null) {
            long val = (long) LoadCurveGenerator.scatter(ev0Value, 0.01); // scatter
            // the
            // value
            c.setValue(2, new SML_CosemValue(new Unsigned64(val)));
        }

        // get the "instanteneous voltage" object
        c = Helper.findObjectInList(new Integer16((short) 3),
                new OctetString("1.0.12.7.0.255"), objList);
        if (c != null) {
            // simulate with 230 Volt, two decimals behind the comma
            int val = (int) LoadCurveGenerator.scatter(23000, 0.001);
            c.setValue(2, new SML_CosemValue(new Integer32(val)));
        }
    }

    /**
     * Retrieves a value from the load profile.
     * 
     * @param index
     * @param profile
     * @return value in mili Wh
     */
    private double getProfileValue(int index, ArrayList<Double> profile) {
        if (index >= profile.size() || index < 0)
            return 0;
        return profile.get(index) * Math.pow(10, 6); // convert (mili Wh: *10^6)
                                                     // to get all decimals
                                                     // behind the comma from
                                                     // the loadprofile
    }

    /**
     * Simulates the current profile index.
     * 
     * @param secondsPerQuarterHour
     * @param profile
     * @return
     */
    private int simulateProfileIndex(int secondsPerQuarterHour,
            ArrayList<Double> profile) {
        int factor = secondsPerQuarterHour * 1000; // convert to milliseconds
        return (int) ((Helper.getTime() / factor) % profile.size()); // modulo
                                                                     // magic
    }

    /**
     * Scatters values randomly (uniform) around an average value, with a
     * maximum variance.
     * 
     * @param mean
     * @param variance
     *            < 1 & > 0
     * @return
     */
    public static double scatter(double mean, double variance) {
        return (mean + ((Math.random() - 0.5) * 2) * (mean * variance));
    }

    /**
     * Runs the simulation.
     */
    @Override
    public void run() {
        while (!isInterrupted()) {
            // simulate
            if (serverModel.getLdev(new OctetString("1")) != null) {
                simulateNextStep(serverModel.getLdev(new OctetString("1"))
                        .getObjectList()); // simulate only MLD (sap always 1)
            }
            // take a nap...
            try {
                Thread.sleep(simulationResolution);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }

}
