/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.smartmeter.ui;

import java.io.IOException;
import java.util.Scanner;

import org.openmuc.jsml.structures.OctetString;
import org.openmuc.jsml.tl.SML_TSAP;

import de.ekut.informatik.kn.josef.mycosem.ServerModel;
import de.ekut.informatik.kn.josef.smartmeter.config.SimulatorConfigType;
import de.ekut.informatik.kn.josef.smartmeter.cosem.RequestHandler;
import de.ekut.informatik.kn.josef.smartmeter.cosem.Server;
import de.ekut.informatik.kn.josef.smartmeter.cosem.SmartMeterModel;
import de.ekut.informatik.kn.josef.smartmeter.simulation.LoadCurveGenerator;

/**
 * The command line interface for the smart meter Simulator. Currently, this
 * class also encapsulates parts of the simulator's business logic. This may
 * change in future releases.
 * 
 * @author Daniel Fuchs
 * @author Michael Hoefling
 */
public class CommandLineInterface {

    /**
     * Configuration.
     */
    private SimulatorConfigType cfg;

    /**
     * COSEM Server model.
     */
    private ServerModel serverModel;

    /**
     * Connection Listener for incoming connections.
     */
    private Server connectionListener;

    /**
     * Smart Meter Simulator.
     */
    private LoadCurveGenerator simulator;

    /**
     * Access point to jSML library.
     */
    private SML_TSAP tsap;

    /**
     * Constructor for the command line interface. A valid configuration must
     * always be passed.
     * 
     * @param cfg
     *            configuration
     */
    public CommandLineInterface(SimulatorConfigType cfg) {
        this.cfg = cfg;
        initConfig();
    }

    /**
     * Initializes the configuration and starts the components.
     * 
     * @param cfg
     */
    private void initConfig() {
        serverModel = new SmartMeterModel();

        RequestHandler handler;
        OctetString username = null;
        OctetString password = null;
        if (cfg.getUsername() != null && cfg.getPassword() != null) {
            username = new OctetString(cfg.getUsername());
            password = new OctetString(cfg.getPassword());
            handler = new RequestHandler(username, password); // with
                                                              // authentication
            System.out
                    .println("Username and password set, use authentication.");
        } else {
            // optional: set username and password = null.
            handler = new RequestHandler(); // without authentication
            System.out.println(
                    "Username and password remain empty, don't use authentication.");
        }

        handler.setServerModel(serverModel);
        connectionListener = new Server(handler);

        simulator = new LoadCurveGenerator(serverModel,
                cfg.getSecPerQuarterHour());
        simulator.setAnnualH0(cfg.getAnnualConsumption());
        simulator.setAnnualEV0(cfg.getAnnualInfeed());
        simulator.start();

        tsap = new SML_TSAP(cfg.getPort(), connectionListener);
        try {
            tsap.startListening();
        } catch (IOException e) {
            // if not possible to listen on port, exit
            System.err.println("An error occurred during socket creation: "
                    + e.getMessage());
            System.exit(0);
        }
        System.out.println(
                "Smart Meter Simulator configured: " + cfg.printPretty());
        System.out.println("Smart Meter Simulator started listening on port: "
                + cfg.getPort() + "...");
    }

    /**
     * Runs the menu, waiting for user input.
     */
    public void runCLI() {
        System.out.println(
                "\nWelcome to the Smart Meter Simulator user interface!");
        Scanner scanner = new Scanner(System.in);
        while (true) {
            System.out.println("Menu:");
            System.out.println("[1] View server settings.");
            System.out.println("[2] View simulator settings.");
            System.out.println("[0] Exit.");
            String input = scanner.next();

            switch (Integer.valueOf(input)) {
            case 0:
                closeSmartMeterSimulator(scanner);
                break;
            case 1:
                viewServerSettings();
                break;
            case 2:
                viewSimulatorSettings();
                break;
            default:
                break;
            }
        }
    }

    /**
     * Close the program.
     */
    private void closeSmartMeterSimulator(Scanner scanner) {
        System.out.println("Closing program...");
        scanner.close();
        System.exit(0);
    }

    /**
     * View the server settings.
     */
    private void viewServerSettings() {
        System.out.println("View server settings:");
        System.out.println(
                "-> Server is listening on port [" + cfg.getPort() + "]");
        if (cfg.getUsername() != null && cfg.getPassword() != null) {
            System.out.println(
                    "-> Server is using authentication for a client [[Username:"
                            + cfg.getUsername() + "][Password:"
                            + cfg.getPassword() + "]]");
        } else {
            System.out.println(
                    "-> Server is not using authentication for clients");
        }
    }

    /**
     * View the simulator settings.
     */
    private void viewSimulatorSettings() {
        System.out.println("View simulator settings:");
        System.out.println("-> Simulated annual power consumption in kWh ["
                + simulator.getAnnualH0() + "]");
        System.out.println("-> Simulated annual power infeed in kWh ["
                + simulator.getAnnualEV0() + "]");
        System.out.println(
                "-> Simulation-time of a 'realtime-quarter-hour' in seconds ["
                        + simulator.getSecPerQuarterHour() + "]");
        System.out.println("-> Simulation resolution in miliseconds ["
                + simulator.getSimulationResolution() + "]");
    }

}
