/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.smartmetergateway.cosem;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.openmuc.jsml.structures.OctetString;
import org.openmuc.jsml.structures.SML_AttentionRes;
import org.openmuc.jsml.structures.SML_File;
import org.openmuc.jsml.structures.SML_Message;
import org.openmuc.jsml.structures.SML_PublicOpenRes;
import org.openmuc.jsml.tl.SML_TConnection;
import org.openmuc.jsml.tl.SML_TSAP;

import de.ekut.informatik.kn.josef.helper.Helper;
import de.ekut.informatik.kn.josef.helper.SMLMessageCreator;
import de.ekut.informatik.kn.josef.mycosem.InterfaceClass;
import de.ekut.informatik.kn.josef.mycosem.LogicalDevice;
import de.ekut.informatik.kn.josef.mycosem.ServerModel;
import de.ekut.informatik.kn.josef.myjsml.SML_COSEM_MessageBody;
import de.ekut.informatik.kn.josef.myjsml.SML_ActionCosemRes;
import de.ekut.informatik.kn.josef.myjsml.SML_GetCosemRes;
import de.ekut.informatik.kn.josef.myjsml.SML_SetCosemRes;
import de.ekut.informatik.kn.josef.smartmetergateway.restclient.RestClient;

/**
 * Class that represents a COSEM client that is communicating with a COSEM
 * server. There is one instance of this class per client-server pair.
 * 
 * @author Daniel Fuchs
 * @author Michael Hoefling
 */
public class Client extends Thread {

    /**
     * Counter for the transaction ID in the SML communication. Usage only
     * through the corresponding method!
     */
    private long transactionIDCounter = 0;

    /**
     * List with all planned requests (like a queue). A Request can be GET, SET
     * or ACTION. Usage only through the corresponding method!
     */
    private ArrayList<CosemRequest> requestList;

    /**
     * Filter rules determing which requests are automatically generated.
     */
    private Filter filter;

    /**
     * Remodels the state of a logical server on the client-side.
     */
    private ServerModel serverModel;

    /**
     * Response Handler. Could also be static.
     */
    private ResponseHandler responseHandler;

    /**
     * ID of this client. Used as COSEM logical client sap.
     */
    private OctetString clientID;

    /**
     * ID of the server. Used as COSEM logical device sap.
     */
    private OctetString serverID;

    /**
     * User's name.
     */
    private OctetString username;

    /**
     * User's password.
     */
    private OctetString password;

    /**
     * Server IP address.
     */
    private String serverIPAddress;

    /**
     * Server port.
     */
    private int serverPort;

    /**
     * Server access point via jSML library.
     */
    private SML_TSAP serverTSAP;

    /**
     * Server connection via jSML library.
     */
    private SML_TConnection serverConnection;

    /**
     * RESTful client.
     */
    private RestClient restClient;

    /**
     * Interval of the thread in milliseconds. Default is 2000.
     */
    private long interval = 2000;

    /**
     * Is the client sending automatic requests?
     */
    private boolean isSendingAuto = true;

    /**
     * Constructor.
     */
    public Client() {
    }

    /**
     * Adds the client ID.
     * 
     * @param clientID
     */
    private void setClientID(OctetString clientID) {
        this.clientID = clientID;
    }

    /**
     * Adds a server to this client.
     * 
     * @param serverID
     * @param serverIPAddress
     * @param serverPort
     */
    private void setServer(OctetString serverID, String serverIPAddress,
            int serverPort) {
        this.serverID = serverID;
        this.serverIPAddress = serverIPAddress;
        this.serverPort = serverPort;
    }

    /**
     * Adds a user with password to this client.
     * 
     * @param username
     * @param password
     */
    private void setUser(OctetString username, OctetString password) {
        this.username = username;
        this.password = password;
    }

    /**
     * Sets the thread interval in miliseconds.
     * 
     * @param ms
     */
    public void setInterval(long ms) {
        this.interval = ms;
    }

    /**
     * Sets the automatic sending of requests.
     * 
     * @param b
     */
    public void setIsSendingAuto(boolean b) {
        this.isSendingAuto = b;
    }

    /**
     * Gets the filter. Necessary for the GUI.
     * 
     * @return
     */
    public Filter getFilter() {
        return this.filter;
    }

    /**
     * Gets the REST client. Necessary for the GUI.
     * 
     * @return
     */
    public RestClient getRestClient() {
        return this.restClient;
    }

    /**
     * Gets the request list of this client. For thread-safety, only access via
     * this getter.
     * 
     * @return
     */
    private synchronized ArrayList<CosemRequest> getRequestList() {
        return this.requestList;
    }

    /**
     * Adds a manual request to the request list.
     * 
     * @param request
     */
    public void addManualRequest(CosemRequest request) {
        getRequestList().add(request);
    }

    /**
     * Gets the object list of the connected server.
     * 
     * @return
     */
    public ArrayList<InterfaceClass> getServerObjectList() {
        if (serverModel.getLdev(serverID) == null) {
            return new ArrayList<InterfaceClass>();
        } else {
            return serverModel.getLdev(serverID).getObjectList();
        }
    }

    /**
     * Handles the sending of SML Files. Creates SML Messages based on the
     * request list and collects them in one SML Request File. Receives the SML
     * Response File and forwards it to the response handler.
     */
    private void sendHandler() {
        try {
            // if there is at least one request present, build a SML request
            // file
            if (getRequestList().size() > 0) {
                ByteArrayOutputStream bs = new ByteArrayOutputStream(100);
                DataOutputStream os = new DataOutputStream(bs);
                // at the start, there is always one openRequest
                SML_Message open = SMLMessageCreator.createOpenRequestMessage(
                        getNextTransID(), this.clientID, getNextReqFileID(),
                        this.serverID, this.username, this.password);
                open.code(os);

                int messageCounter = 2; // start with 2, because of open and
                                        // close
                Iterator<CosemRequest> it = getRequestList().iterator(); // for
                                                                         // each
                // request
                // in the
                // list
                while (it.hasNext()) {
                    CosemRequest request = it.next();
                    SML_Message m = request.createRequestMessage(
                            getNextTransID(), this.clientID, this.serverID,
                            this.username, this.password);
                    if (m != null) {
                        m.code(os);
                        messageCounter++;
                    }
                    it.remove();
                }
                // there is always one closeRequest at the end
                SML_Message close = SMLMessageCreator
                        .createCloseRequestMessage(getNextTransID());
                close.code(os);

                // SML file is ready to send
                byte[] requestBuffer = bs.toByteArray();
                this.serverConnection = this.serverTSAP.connectTo(
                        InetAddress.getByName(this.serverIPAddress),
                        this.serverPort, 0);
                System.out.println("Sending a SML Request File containing "
                        + messageCounter + " request message(s)...");
                this.serverConnection.send(requestBuffer);
                SML_File smlFile = this.serverConnection.receive(); // receive
                                                                    // the
                                                                    // response
                                                                    // file.
                handleSMLResponseFile(smlFile); // handle response file

            } else {
                System.out.println("No request in list -> do nothing.");
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * Handles a SML response file.
     * 
     * @param smlFile
     */
    private void handleSMLResponseFile(SML_File smlFile) {
        // jSML checks already validity during the en/decoding.
        List<SML_Message> smlMessages = smlFile.getMessages();
        System.out.println("Received a SML Response File containing "
                + smlMessages.size() + " messages.");

        for (SML_Message message : smlMessages) {
            int tag = message.getMessageBody().getTag().getVal();
            String transID = message.getTransactionId().toString();
            switch (tag) {

            case SML_COSEM_MessageBody.OpenResponse:
                OctetString reqFileID = ((SML_PublicOpenRes) message
                        .getMessageBody().getChoice()).getReqFileId();
                System.out.println("Received openResponse [TransactionID:"
                        + transID + ", FileID:" + reqFileID + "]");
                break;

            case SML_COSEM_MessageBody.GetCosemResponse:
                System.out.println("Received getCosemResponse [TransactionID:"
                        + transID + "]");
                responseHandler.cosemGetResponseHandler(
                        (SML_GetCosemRes) message.getMessageBody().getChoice());
                break;

            case SML_COSEM_MessageBody.SetCosemResponse:
                System.out.println("Received setCosemResponse [TransactionID:"
                        + transID + "]");
                responseHandler.cosemSetResponseHandler(
                        (SML_SetCosemRes) message.getMessageBody().getChoice());
                break;

            case SML_COSEM_MessageBody.ActionCosemResponse:
                System.out
                        .println("Received actionCosemResponse [TransactionID:"
                                + transID + "]");
                responseHandler
                        .cosemActionResponseHandler((SML_ActionCosemRes) message
                                .getMessageBody().getChoice());
                break;

            case SML_COSEM_MessageBody.CloseResponse:
                System.out.println("Received closeResponse [TransactionID:"
                        + transID + "]");
                System.out.println("");
                break;

            case SML_COSEM_MessageBody.AttentionResponse:
                System.out.println("Received attentionResponse [TransactionID:"
                        + transID + "]");
                responseHandler
                        .attentionResponseHandler((SML_AttentionRes) message
                                .getMessageBody().getChoice());
                break;

            default:
                System.out.println("Received non-response message: discard.");
            }
        }
    }

    /**
     * Generates an association view request: current AssociationLN object, with
     * index 2 for object list. Adds the request to the requestList. Creates the
     * remodeled logical device, in which the objects will be stored.
     */
    private void addAssociationViewRequest() {
        GetRequest request = new GetRequest("0.0.40.0.0.255", (short) 15,
                (short) 1, new short[] { (short) 2 });
        GetRequest requestName = new GetRequest("0.0.42.0.0.255", (short) 1,
                (short) 0);
        getRequestList().add(request);
        getRequestList().add(requestName);
    }

    /**
     * Gets the next transaction ID for SML communication. Needed to map
     * corresponding request messages to response messages. Uniquely per
     * server/client setup.
     * 
     * @return transactionID
     */
    private OctetString getNextTransID() { // parameter: server seems nice, if
                                           // multiple servers
        this.transactionIDCounter++;
        return new OctetString(this.transactionIDCounter + ""); // + server
                                                                // seems nice
    }

    /**
     * Gets the next request file ID. Needed to map corresponding request files
     * to response files. Uniquely in the system.
     * 
     * @return reqFileID
     */
    private OctetString getNextReqFileID() {
        return new OctetString(Helper.getTime() + "");
    }

    /**
     * Cosem client thread. Invokes the handler to send requests in regular
     * intervals.
     */
    @Override
    public void run() {
        while (!isInterrupted()) {
            // send handler
            sendHandler();
            // automatic request handler
            if (isSendingAuto) {
                getRequestList().addAll(filter.applyFilterRules(
                        serverModel.getLdev(serverID).getObjectList()));
            }
            // take a nap...
            try {
                Thread.sleep(interval);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    };

    /**
     * Initializes and starts a client. Only run once.
     * 
     * @param clientID
     * @param serverID
     * @param serverIP
     * @param port
     * @param username
     * @param password
     */
    public void initialize(String clientID, String serverID, String serverIP,
            int port, String username, String password) {
        setClientID(new OctetString(clientID));
        setServer(new OctetString(serverID), serverIP, port);
        setUser(new OctetString(username), new OctetString(password));

        requestList = new ArrayList<CosemRequest>();
        filter = new Filter();
        serverModel = new ServerModel();
        serverModel.getLdevs()
                .add(new LogicalDevice(new OctetString(serverID)));
        responseHandler = new ResponseHandler(serverModel);
        serverTSAP = new SML_TSAP();
        restClient = new RestClient(serverModel);

        addAssociationViewRequest(); // request the server's object model

        start();

        System.out.println("Cosem client initialized and started.");
    }

    @Override
    public void interrupt() {
        this.restClient.endThread();
        super.interrupt();
    }

}
