/*
 * jOSEF: A Java-Based Open-Source Smart Meter Gateway Experimentation Framework
 *
 * Copyright (C) 2015 Daniel Fuchs
 * Copyright (C) 2015 Michael Hoefling
 *
 * jOSEF is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jOSEF is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cobertura; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

package de.ekut.informatik.kn.josef.smartmetergateway.ui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JPanel;
import javax.swing.Timer;

import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.XYPlot;
import org.jfree.data.time.Millisecond;
import org.jfree.data.time.TimeSeries;
import org.jfree.data.time.TimeSeriesCollection;
import org.jfree.data.xy.XYDataset;

import de.ekut.informatik.kn.josef.helper.Helper;
import de.ekut.informatik.kn.josef.mycosem.InterfaceClass;
import de.ekut.informatik.kn.josef.mycosem.Register;
import de.ekut.informatik.kn.josef.mycosem.ScalerUnit;

/**
 * Class that draws a chart on a JPanel, based on the values of a COSEM object.
 * The drawing is realized with the JFreeChart library. Inspired by:
 * http://dirtyhandsphp.blogspot.in/2012/07/how-to-draw-dynamic-line-or-timeseries.html
 * 
 * @author Daniel Fuchs
 */
public class CosemObjectChartPanel extends JPanel implements ActionListener {

    /**
     * Serial.
     */
    private static final long serialVersionUID = 1L;

    /**
     * Basic color for the background.
     */
    private Color basicColor = new Color(0xe8e8e8);

    /**
     * The time series.
     */
    private TimeSeries series;

    /**
     * The last added value.
     */
    private Number lastValue;

    /**
     * Timer to refresh the chart diagram.
     */
    private Timer timer = new Timer(250, this);

    /**
     * Reference to the COSEM object.
     */
    private InterfaceClass object;

    /**
     * Constructor.
     * 
     * @param title
     */
    public CosemObjectChartPanel(String title) {
        this.setBackground(basicColor);
    }

    /**
     * Draws a new chart on the panel.
     * 
     * @param title
     * @param legend
     * @param objectName
     * @param objectClassID
     * @param objectList
     */
    @SuppressWarnings("deprecation")
    public void drawNewChart(InterfaceClass ic) {
        reset(); // clear
        if (ic == null)
            return;

        this.object = ic;
        this.series = new TimeSeries("Value of attribute 2", Millisecond.class);
        TimeSeriesCollection dataset = new TimeSeriesCollection(this.series);
        String title = "Object with logical name ["
                + ic.getLogicalName().toString() + "] and class id ["
                + ic.getClassID().getVal() + "]";
        JFreeChart chart = createChart(dataset, title);
        // chart.setTitle(new TextTitle(title, new Font(Font.SERIF, Font.PLAIN,
        // 12)));

        chart.setBackgroundPaint(basicColor);
        this.setLayout(new BorderLayout());
        ChartPanel chartPanel = new ChartPanel(chart);
        this.add(chartPanel);

        timer.setInitialDelay(1000);
        timer.start();
    }

    /**
     * Resets the panel.
     */
    public void reset() {
        this.timer.stop();
        this.removeAll();
    }

    /**
     * Creates the chart.
     * 
     * @param dataset
     * @param title
     * @return
     */
    private JFreeChart createChart(XYDataset dataset, String title) {
        JFreeChart chart = ChartFactory.createTimeSeriesChart(title, "Time",
                "Value", dataset, true, true, false);

        XYPlot plot = chart.getXYPlot();
        plot.setBackgroundPaint(new Color(0xffffe0)); // yellow
        plot.setBackgroundAlpha(0.5f);
        plot.setDomainGridlinesVisible(true);
        plot.setDomainGridlinePaint(Color.LIGHT_GRAY);
        plot.setRangeGridlinesVisible(true);
        plot.setRangeGridlinePaint(Color.LIGHT_GRAY);

        // plot.getRenderer().setPaint(Color.BLUE); // ok

        ValueAxis xaxis = plot.getDomainAxis();
        xaxis.setAutoRange(true);
        xaxis.setFixedAutoRange(60000.0); // visible range is 60 seconds
        xaxis.setVerticalTickLabels(true);

        ValueAxis yaxis = plot.getRangeAxis();
        yaxis.setAutoRange(true);

        return chart;
    }

    /**
     * On action, retrieve current value from object and add it to the time
     * series. This method is periodically called. Is able to draw Data and
     * Register, if the value is a number.
     */
    public void actionPerformed(final ActionEvent e) {
        // IC Data
        if (object instanceof de.ekut.informatik.kn.josef.mycosem.Data) {
            Number n = Helper.getNumberValue(object.getValue(2));
            this.lastValue = n;
            this.series.add(new Millisecond(), this.lastValue);
        }
        // IC Register, try to scale via scaler_unit
        if (object instanceof de.ekut.informatik.kn.josef.mycosem.Register) {
            Number n = Helper.getNumberValue(object.getValue(2));
            this.lastValue = n;
            try { // try scaling
                Register reg = (Register) object;
                ScalerUnit su = reg.getScalerUnit();
                int scaler = su.getScaler().getVal();
                int unit = su.getUnit().getVal();
                double val = n.doubleValue();
                val = val * Math.pow(10, scaler);
                this.lastValue = val;

                String unitString = ""; // interpret the unit
                if (unit == 27)
                    unitString = "(Watt)";
                if (unit == 30)
                    unitString = "(Watt hour)";
                if (unit == 35)
                    unitString = "(Volt)";
                this.series.setKey("Value of attribute 2 in unit [" + unit
                        + "] " + unitString);
            } catch (Exception e2) {
            }
            this.series.add(new Millisecond(), this.lastValue);
        }
    }

}
