#####################
# This R script can be used to evaluate the data of the first experiment.
# You only need to change the working directory to the path you extracted
# the data to. Further, two additional packages are required:
# ez (performs the actual ANOVAs)
# schoRsch (allows convenient formatting of the ANOVA tables)
# You can install these packages via the shell (internet connection required):
# install.packages("ez", dependencies = TRUE)
# install.packages("schoRsch", dependencies = TRUE)
#
# The data contains two dependent measures: Response times and error rates.
# The analysis is separated for change and no-change trials. Both for change
# and no-change trials, there are two different analysis: The main analysis,
# the analysis of trials which required no mental rotation. For the change
# trials, there is also an analysis considering the different rotation disparities.
#####################

library("ez")
library("schoRsch")

# enter the working directory here
setwd("")


##################### Change Trials: Main Analysis

# obtain the data...
data <- read.table("ANOVA_Data.dat", header = TRUE)
# the vectors contain the factor levels, you can use them to filter the data set
usedParity             <- c("canon", "mirror")
usedTactileRotDir      <- c("clockwise", "clockwise_to_counterclockwise", "counterclockwise", "counterclockwise_to_clockwise")
# change only
usedTactileRotDir      <- c("clockwise_to_counterclockwise", "counterclockwise_to_clockwise")
usedMentalRotDir       <- c("none", "clockwise", "counterclockwise")
# only with rotation
usedMentalRotDir       <- c("clockwise", "counterclockwise")
usedDisparity          <- c("0DEG", "070DEG", "100DEG", "130DEG", "160DEG")
usedLetterIdentities   <- c("R", "L", "F", "P")
# apply the filters: only trials with changes in the tactile stimulation and with mental rotation
data <- subset(data, TactileRotDir %in% usedTactileRotDir)
data <- subset(data, MentalRotDir %in% usedMentalRotDir)
# aggregate data and remove empty cells
data <- aggregate(cbind(RT,ER) ~ ParticipantID*Parity*TactileRotDir*MentalRotDir, data, mean)
data <- droplevels(data);

summary(data)

data$ParticipantID <- factor(data$ParticipantID)

# the actual ANOVAs go here, these are tables 3 and 4 in the manuscript

# RT ANOVA
anova_rt <- ezANOVA(data=data, dv = .(RT), wid = .(ParticipantID), within = .(Parity,TactileRotDir,MentalRotDir), detailed = TRUE)
anova_out(anova_rt)
# error rate analysis
anova_er <- ezANOVA(data=data, dv = .(ER), wid = .(ParticipantID), within = .(Parity,TactileRotDir,MentalRotDir), detailed = TRUE)
anova_out(anova_er)

# post-hoc t-test analysis for the response times, tests are two-sided, p-values are adjusted according to the method of Holm

# interaction between parity and tactile rotation direction
data_aggregated <- aggregate(RT ~ ParticipantID*Parity*TactileRotDir, data, mean)
t1 = t.test(data_aggregated[data_aggregated$Parity == "mirror"   & data_aggregated$TactileRotDir == "clockwise_to_counterclockwise",]$RT, data_aggregated[data_aggregated$Parity == "mirror"   & data_aggregated$TactileRotDir == "counterclockwise_to_clockwise",]$RT, paired = TRUE, alternative = 'two.sided')
t2 = t.test(data_aggregated[data_aggregated$Parity == "canon"    & data_aggregated$TactileRotDir == "clockwise_to_counterclockwise",]$RT, data_aggregated[data_aggregated$Parity == "canon"    & data_aggregated$TactileRotDir == "counterclockwise_to_clockwise",]$RT, paired = TRUE, alternative = 'two.sided')
t3 = t.test(data_aggregated[data_aggregated$Parity == "mirror"   & data_aggregated$TactileRotDir == "clockwise_to_counterclockwise",]$RT, data_aggregated[data_aggregated$Parity == "canon"    & data_aggregated$TactileRotDir == "clockwise_to_counterclockwise",]$RT, paired = TRUE, alternative = 'two.sided')
t4 = t.test(data_aggregated[data_aggregated$Parity == "canon"    & data_aggregated$TactileRotDir == "counterclockwise_to_clockwise",]$RT, data_aggregated[data_aggregated$Parity == "mirror"   & data_aggregated$TactileRotDir == "counterclockwise_to_clockwise",]$RT, paired = TRUE, alternative = 'two.sided')

format(p.adjust(c(t1$p.value, t2$p.value, t3$p.value, t4$p.value), method = "holm"), scientific = FALSE)
format(c(t1$p.value, t2$p.value, t3$p.value, t4$p.value), scientific = FALSE)

# interaction between parity and mental rotation direction
data_aggregated <- aggregate(RT ~ ParticipantID*Parity*MentalRotDir, data, mean)
t1 = t.test(data_aggregated[data_aggregated$Parity == "mirror"   & data_aggregated$MentalRotDir == "clockwise",]$RT, data_aggregated[data_aggregated$Parity == "mirror"   & data_aggregated$MentalRotDir == "counterclockwise",]$RT, paired = TRUE, alternative = 'two.sided')
t2 = t.test(data_aggregated[data_aggregated$Parity == "canon"    & data_aggregated$MentalRotDir == "clockwise",]$RT, data_aggregated[data_aggregated$Parity == "canon"    & data_aggregated$MentalRotDir == "counterclockwise",]$RT, paired = TRUE, alternative = 'two.sided')
t3 = t.test(data_aggregated[data_aggregated$Parity == "mirror"   & data_aggregated$MentalRotDir == "clockwise",]$RT, data_aggregated[data_aggregated$Parity == "canon"    & data_aggregated$MentalRotDir == "clockwise",]$RT, paired = TRUE, alternative = 'two.sided')
t4 = t.test(data_aggregated[data_aggregated$Parity == "canon"    & data_aggregated$MentalRotDir == "counterclockwise",]$RT, data_aggregated[data_aggregated$Parity == "mirror"   & data_aggregated$MentalRotDir == "counterclockwise",]$RT, paired = TRUE, alternative = 'two.sided')

format(p.adjust(c(t1$p.value, t2$p.value, t3$p.value, t4$p.value), method = "holm"), scientific = FALSE)
format(c(t1$p.value, t2$p.value, t3$p.value, t4$p.value), scientific = FALSE)

# interaction between mental and tactile rotation direction
data_aggregated <- aggregate(RT ~ ParticipantID*TactileRotDir*MentalRotDir, data, mean)
t1 = t.test(data_aggregated[data_aggregated$TactileRotDir == "clockwise_to_counterclockwise"   & data_aggregated$MentalRotDir == "clockwise",]$RT, data_aggregated[data_aggregated$TactileRotDir == "clockwise_to_counterclockwise"   & data_aggregated$MentalRotDir == "counterclockwise",]$RT, paired = TRUE, alternative = 'two.sided')
t2 = t.test(data_aggregated[data_aggregated$TactileRotDir == "counterclockwise_to_clockwise"    & data_aggregated$MentalRotDir == "clockwise",]$RT, data_aggregated[data_aggregated$TactileRotDir == "counterclockwise_to_clockwise"    & data_aggregated$MentalRotDir == "counterclockwise",]$RT, paired = TRUE, alternative = 'two.sided')
t3 = t.test(data_aggregated[data_aggregated$TactileRotDir == "clockwise_to_counterclockwise"   & data_aggregated$MentalRotDir == "clockwise",]$RT, data_aggregated[data_aggregated$TactileRotDir == "counterclockwise_to_clockwise"    & data_aggregated$MentalRotDir == "clockwise",]$RT, paired = TRUE, alternative = 'two.sided')
t4 = t.test(data_aggregated[data_aggregated$TactileRotDir == "counterclockwise_to_clockwise"    & data_aggregated$MentalRotDir == "counterclockwise",]$RT, data_aggregated[data_aggregated$TactileRotDir == "clockwise_to_counterclockwise"   & data_aggregated$MentalRotDir == "counterclockwise",]$RT, paired = TRUE, alternative = 'two.sided')

format(p.adjust(c(t1$p.value, t2$p.value, t3$p.value, t4$p.value), method = "holm"), scientific = FALSE)
format(c(t1$p.value, t2$p.value, t3$p.value, t4$p.value), scientific = FALSE)


# post-hoc t-test analysis for the error rates, tests are two-sided, p-values are adjusted according to the method of Holm

# interaction between parity and mental rotation direction
data_aggregated <- aggregate(ER ~ ParticipantID*Parity*MentalRotDir, data, mean)
t1 = t.test(data_aggregated[data_aggregated$Parity == "canon"  & data_aggregated$MentalRotDir == "clockwise",]$ER, data_aggregated[data_aggregated$Parity == "canon"  & data_aggregated$MentalRotDir == "counterclockwise",]$ER, paired = TRUE, alternative = 'two.sided')
t2 = t.test(data_aggregated[data_aggregated$Parity == "mirror" & data_aggregated$MentalRotDir == "clockwise",]$ER, data_aggregated[data_aggregated$Parity == "mirror" & data_aggregated$MentalRotDir == "counterclockwise",]$ER, paired = TRUE, alternative = 'two.sided')

format(p.adjust(c(t1$p.value, t2$p.value), method = "holm"), scientific = FALSE)
format(c(t1$p.value, t2$p.value), scientific = FALSE)

# interaction between mental and tactile rotation direction
data_aggregated <- aggregate(ER ~ ParticipantID*TactileRotDir*MentalRotDir, data, mean)
t1 = t.test(data_aggregated[data_aggregated$TactileRotDir == "clockwise_to_counterclockwise"   & data_aggregated$MentalRotDir == "clockwise",]$ER, data_aggregated[data_aggregated$TactileRotDir == "counterclockwise_to_clockwise"    & data_aggregated$MentalRotDir == "clockwise",]$ER, paired = TRUE, alternative = 'two.sided')
t2 = t.test(data_aggregated[data_aggregated$TactileRotDir == "counterclockwise_to_clockwise"    & data_aggregated$MentalRotDir == "counterclockwise",]$ER, data_aggregated[data_aggregated$TactileRotDir == "clockwise_to_counterclockwise"   & data_aggregated$MentalRotDir == "counterclockwise",]$ER, paired = TRUE, alternative = 'two.sided')

format(p.adjust(c(t1$p.value, t2$p.value), method = "holm"), scientific = FALSE)
format(c(t1$p.value, t2$p.value), scientific = FALSE)

# interaction between parity and tactile rotation direction
data_aggregated <- aggregate(ER ~ ParticipantID*TactileRotDir*Parity, data, mean)
t1 = t.test(data_aggregated[data_aggregated$TactileRotDir == "clockwise_to_counterclockwise"   & data_aggregated$Parity == "mirror",]$ER, data_aggregated[data_aggregated$TactileRotDir == "counterclockwise_to_clockwise"    & data_aggregated$Parity == "mirror",]$ER, paired = TRUE, alternative = 'two.sided')
t2 = t.test(data_aggregated[data_aggregated$TactileRotDir == "counterclockwise_to_clockwise"    & data_aggregated$Parity == "canon",]$ER, data_aggregated[data_aggregated$TactileRotDir == "clockwise_to_counterclockwise"   & data_aggregated$Parity == "canon",]$ER, paired = TRUE, alternative = 'two.sided')

format(p.adjust(c(t1$p.value, t2$p.value), method = "holm"), scientific = FALSE)
format(c(t1$p.value, t2$p.value), scientific = FALSE)

# threeway interaction between parity, mental rotation direction and tactile rotation direction

tests = pairwise.t.test(data$ER, interaction(data$Parity,data$TactileRotDir,data$MentalRotDir), p.adj = "holm", pool.sd = FALSE, paired = TRUE)
tests$p.value

data_aggregated <- aggregate(ER ~ ParticipantID*Parity*TactileRotDir*MentalRotDir, data, mean)
t1 = t.test(data_aggregated[data_aggregated$TactileRotDir == "counterclockwise_to_clockwise" & data_aggregated$MentalRotDir == "clockwise"        & data_aggregated$Parity  == "canon",]$ER, data_aggregated[data_aggregated$TactileRotDir == "clockwise_to_counterclockwise" & data_aggregated$MentalRotDir == "clockwise"         & data_aggregated$Parity  == "canon",]$ER, paired = TRUE, alternative = 'two.sided')
t2 = t.test(data_aggregated[data_aggregated$TactileRotDir == "counterclockwise_to_clockwise" & data_aggregated$MentalRotDir == "counterclockwise" & data_aggregated$Parity  == "canon",]$ER, data_aggregated[data_aggregated$TactileRotDir == "clockwise_to_counterclockwise" & data_aggregated$MentalRotDir == "counterclockwise"  & data_aggregated$Parity  == "canon",]$ER, paired = TRUE, alternative = 'two.sided')
t3 = t.test(data_aggregated[data_aggregated$TactileRotDir == "counterclockwise_to_clockwise" & data_aggregated$MentalRotDir == "clockwise"        & data_aggregated$Parity  == "mirror",]$ER, data_aggregated[data_aggregated$TactileRotDir == "clockwise_to_counterclockwise" & data_aggregated$MentalRotDir == "clockwise"        & data_aggregated$Parity  == "mirror",]$ER, paired = TRUE, alternative = 'two.sided')
t4 = t.test(data_aggregated[data_aggregated$TactileRotDir == "counterclockwise_to_clockwise" & data_aggregated$MentalRotDir == "counterclockwise" & data_aggregated$Parity  == "mirror",]$ER, data_aggregated[data_aggregated$TactileRotDir == "clockwise_to_counterclockwise" & data_aggregated$MentalRotDir == "counterclockwise" & data_aggregated$Parity  == "mirror",]$ER, paired = TRUE, alternative = 'two.sided')

format(p.adjust(c(t1$p.value, t2$p.value, t3$p.value, t4$p.value), method = "holm"), scientific = FALSE)
format(c(t1$p.value, t2$p.value, t3$p.value, t4$p.value), scientific = FALSE)

##################### Change Trials: Trials without mental rotation
# obtain data
data <- read.table("ANOVA_Data.dat", header = TRUE)

usedParity             <- c("canon", "mirror")
usedTactileRotDir      <- c("clockwise", "clockwise_to_counterclockwise", "counterclockwise", "counterclockwise_to_clockwise")
# change only
usedTactileRotDir      <- c("clockwise_to_counterclockwise", "counterclockwise_to_clockwise")
usedMentalRotDir       <- c("none", "clockwise", "counterclockwise")
# only without rotation
usedMentalRotDir       <- c("none")
usedDisparity          <- c("0DEG", "070DEG", "100DEG", "130DEG", "160DEG")
usedLetterIdentities   <- c("R", "L", "F", "P")
# apply filter
data <- subset(data, TactileRotDir %in% usedTactileRotDir)
data <- subset(data, MentalRotDir %in% usedMentalRotDir)

# aggregate data and remove empty cells
data <- aggregate(cbind(RT,ER) ~ ParticipantID*Parity*TactileRotDir*MentalRotDir, data, mean)
data <- droplevels(data);

summary(data)

data$ParticipantID <- factor(data$ParticipantID)
# RT ANOVA
anova_rt <- ezANOVA(data=data, dv = .(RT), wid = .(ParticipantID), within = .(Parity,TactileRotDir), detailed = TRUE)
anova_out(anova_rt)
# error rate ANOVA
anova_er <- ezANOVA(data=data, dv = .(ER), wid = .(ParticipantID), within = .(Parity,TactileRotDir), detailed = TRUE)
anova_out(anova_er)

##################### Change Trials: Disparity analysis
# obtain data
data <- read.table("ANOVA_Data_With_Disparity.dat", header = TRUE)

usedParity             <- c("canon", "mirror")
usedTactileRotDir      <- c("clockwise", "clockwise_to_counterclockwise", "counterclockwise", "counterclockwise_to_clockwise")
# change only
usedTactileRotDir      <- c("clockwise_to_counterclockwise", "counterclockwise_to_clockwise")
usedMentalRotDir       <- c("none", "clockwise", "counterclockwise")
# only with rotation
usedMentalRotDir       <- c("clockwise", "counterclockwise")
usedDisparity          <- c("0DEG", "070DEG", "100DEG", "130DEG", "160DEG")
usedLetterIdentities   <- c("R", "L", "F", "P")
# apply filter
data <- subset(data, TactileRotDir %in% usedTactileRotDir)
data <- subset(data, MentalRotDir %in% usedMentalRotDir)

# aggregate data and remove empty cells
data <- aggregate(cbind(RT,ER) ~ ParticipantID*Parity*TactileRotDir*MentalRotDir*Disparity, data, mean)
data <- droplevels(data);

summary(data)

data$ParticipantID <- factor(data$ParticipantID)
# response time analysis
anova_rt <- ezANOVA(data=data, dv = .(RT), wid = .(ParticipantID), within = .(Parity,TactileRotDir,MentalRotDir,Disparity), detailed = TRUE)
anova_out(anova_rt)
# error rate analysis
anova_er <- ezANOVA(data=data, dv = .(ER), wid = .(ParticipantID), within = .(Parity,TactileRotDir,MentalRotDir,Disparity), detailed = TRUE)
anova_out(anova_er)

# post hoc analysis of the threeway interaction between partiy, mental rotation direction and disparity
data_aggregated <- aggregate(RT ~ ParticipantID*Parity*MentalRotDir*Disparity, data, mean)
t1 = t.test(data_aggregated[data_aggregated$Parity == "canon" & data_aggregated$MentalRotDir == "clockwise" & data_aggregated$Disparity  == "100DEG",]$RT, data_aggregated[data_aggregated$Parity == "canon" & data_aggregated$MentalRotDir == "counterclockwise" & data_aggregated$Disparity  == "100DEG",]$RT, paired = TRUE, alternative = 'two.sided')
t2 = t.test(data_aggregated[data_aggregated$Parity == "mirror" & data_aggregated$MentalRotDir == "clockwise" & data_aggregated$Disparity  == "100DEG",]$RT, data_aggregated[data_aggregated$Parity == "mirror" & data_aggregated$MentalRotDir == "counterclockwise" & data_aggregated$Disparity  == "100DEG",]$RT, paired = TRUE, alternative = 'two.sided')
t3 = t.test(data_aggregated[data_aggregated$Parity == "canon" & data_aggregated$MentalRotDir == "clockwise" & data_aggregated$Disparity  == "130DEG",]$RT, data_aggregated[data_aggregated$Parity == "canon" & data_aggregated$MentalRotDir == "counterclockwise" & data_aggregated$Disparity  == "130DEG",]$RT, paired = TRUE, alternative = 'two.sided')
t4 = t.test(data_aggregated[data_aggregated$Parity == "mirror" & data_aggregated$MentalRotDir == "clockwise" & data_aggregated$Disparity  == "130DEG",]$RT, data_aggregated[data_aggregated$Parity == "mirror" & data_aggregated$MentalRotDir == "counterclockwise" & data_aggregated$Disparity  == "130DEG",]$RT, paired = TRUE, alternative = 'two.sided')
t5 = t.test(data_aggregated[data_aggregated$Parity == "canon" & data_aggregated$MentalRotDir == "clockwise" & data_aggregated$Disparity  == "160DEG",]$RT, data_aggregated[data_aggregated$Parity == "canon" & data_aggregated$MentalRotDir == "counterclockwise" & data_aggregated$Disparity  == "160DEG",]$RT, paired = TRUE, alternative = 'two.sided')
t6 = t.test(data_aggregated[data_aggregated$Parity == "mirror" & data_aggregated$MentalRotDir == "clockwise" & data_aggregated$Disparity  == "160DEG",]$RT, data_aggregated[data_aggregated$Parity == "mirror" & data_aggregated$MentalRotDir == "counterclockwise" & data_aggregated$Disparity  == "160DEG",]$RT, paired = TRUE, alternative = 'two.sided')
t7 = t.test(data_aggregated[data_aggregated$Parity == "canon" & data_aggregated$MentalRotDir == "clockwise" & data_aggregated$Disparity  == "70DEG",]$RT, data_aggregated[data_aggregated$Parity == "canon" & data_aggregated$MentalRotDir == "counterclockwise" & data_aggregated$Disparity  == "70DEG",]$RT, paired = TRUE, alternative = 'two.sided')
t8 = t.test(data_aggregated[data_aggregated$Parity == "mirror" & data_aggregated$MentalRotDir == "clockwise" & data_aggregated$Disparity  == "70DEG",]$RT, data_aggregated[data_aggregated$Parity == "mirror" & data_aggregated$MentalRotDir == "counterclockwise" & data_aggregated$Disparity  == "70DEG",]$RT, paired = TRUE, alternative = 'two.sided')

format(p.adjust(c(t1$p.value, t2$p.value, t3$p.value, t4$p.value, t5$p.value, t6$p.value, t7$p.value, t8$p.value), method = "holm"), scientific = FALSE)
format(c(t1$p.value, t2$p.value, t3$p.value, t4$p.value, t5$p.value, t6$p.value, t7$p.value, t8$p.value), scientific = FALSE)
aggregate(RT ~ Parity*MentalRotDir*Disparity, data, mean)


##################### No-Change Trials: Main Analysis
# obtain the data...
data <- read.table("ANOVA_Data.dat", header = TRUE)
# the vectors contain the factor levels, you can use them to filter the data set
usedParity             <- c("canon", "mirror")
usedTactileRotDir      <- c("clockwise", "clockwise_to_counterclockwise", "counterclockwise", "counterclockwise_to_clockwise")
# no-change only
usedTactileRotDir      <- c("clockwise", "counterclockwise")
usedMentalRotDir       <- c("none", "clockwise", "counterclockwise")
# only with rotation
usedMentalRotDir       <- c("clockwise", "counterclockwise")
usedDisparity          <- c("0DEG", "070DEG", "100DEG", "130DEG", "160DEG")
usedLetterIdentities   <- c("R", "L", "F", "P")
# apply the filters: only trials with changes in the tactile stimulation and with mental rotation
data <- subset(data, TactileRotDir %in% usedTactileRotDir)
data <- subset(data, MentalRotDir %in% usedMentalRotDir)
# aggregate data and remove empty cells
data <- aggregate(cbind(RT,ER) ~ ParticipantID*Parity*TactileRotDir*MentalRotDir, data, mean)
data <- droplevels(data);

summary(data)

data$ParticipantID <- factor(data$ParticipantID)

# the actual ANOVAs go here, these are tables 1 and 2 in the manuscript

# RT ANOVA
anova_rt <- ezANOVA(data=data, dv = .(RT), wid = .(ParticipantID), within = .(Parity,TactileRotDir,MentalRotDir), detailed = TRUE)
anova_out(anova_rt)
# error rate analysis
anova_er <- ezANOVA(data=data, dv = .(ER), wid = .(ParticipantID), within = .(Parity,TactileRotDir,MentalRotDir), detailed = TRUE)
anova_out(anova_er)

# post-hoc t-test analysis for the response times, tests are two-sided, p-values are adjusted according to the method of Holm

# interaction between parity and mental rotation direction
data_aggregated <- aggregate(RT ~ ParticipantID*Parity*MentalRotDir, data, mean)
t1 = t.test(data_aggregated[data_aggregated$Parity == "mirror"   & data_aggregated$MentalRotDir == "clockwise",]$RT, data_aggregated[data_aggregated$Parity == "mirror"   & data_aggregated$MentalRotDir == "counterclockwise",]$RT, paired = TRUE, alternative = 'two.sided')
t2 = t.test(data_aggregated[data_aggregated$Parity == "canon"    & data_aggregated$MentalRotDir == "clockwise",]$RT, data_aggregated[data_aggregated$Parity == "canon"    & data_aggregated$MentalRotDir == "counterclockwise",]$RT, paired = TRUE, alternative = 'two.sided')
t3 = t.test(data_aggregated[data_aggregated$Parity == "mirror"   & data_aggregated$MentalRotDir == "clockwise",]$RT, data_aggregated[data_aggregated$Parity == "canon"    & data_aggregated$MentalRotDir == "clockwise",]$RT, paired = TRUE, alternative = 'two.sided')
t4 = t.test(data_aggregated[data_aggregated$Parity == "canon"    & data_aggregated$MentalRotDir == "counterclockwise",]$RT, data_aggregated[data_aggregated$Parity == "mirror"   & data_aggregated$MentalRotDir == "counterclockwise",]$RT, paired = TRUE, alternative = 'two.sided')


data_aggregated <- aggregate(ER ~ ParticipantID*Parity*MentalRotDir, data, mean)
t1 = t.test(data_aggregated[data_aggregated$Parity == "mirror"   & data_aggregated$MentalRotDir == "clockwise",]$ER, data_aggregated[data_aggregated$Parity == "mirror"   & data_aggregated$MentalRotDir == "counterclockwise",]$ER, paired = TRUE, alternative = 'two.sided')
t2 = t.test(data_aggregated[data_aggregated$Parity == "canon"    & data_aggregated$MentalRotDir == "clockwise",]$ER, data_aggregated[data_aggregated$Parity == "canon"    & data_aggregated$MentalRotDir == "counterclockwise",]$ER, paired = TRUE, alternative = 'two.sided')
t3 = t.test(data_aggregated[data_aggregated$Parity == "mirror"   & data_aggregated$MentalRotDir == "counterclockwise",]$ER, data_aggregated[data_aggregated$Parity == "canon"    & data_aggregated$MentalRotDir == "clockwise",]$ER, paired = TRUE, alternative = 'two.sided')
t4 = t.test(data_aggregated[data_aggregated$Parity == "canon"    & data_aggregated$MentalRotDir == "counterclockwise",]$ER, data_aggregated[data_aggregated$Parity == "mirror"   & data_aggregated$MentalRotDir == "counterclockwise",]$ER, paired = TRUE, alternative = 'two.sided')

format(p.adjust(c(t1$p.value, t2$p.value, t3$p.value, t4$p.value), method = "holm"), scientific = FALSE)
format(c(t1$p.value, t2$p.value, t3$p.value, t4$p.value), scientific = FALSE)

##################### No-Change Trials: Trials without mental rotation
# obtain data
data <- read.table("ANOVA_Data.dat", header = TRUE)

usedParity             <- c("canon", "mirror")
usedTactileRotDir      <- c("clockwise", "clockwise_to_counterclockwise", "counterclockwise", "counterclockwise_to_clockwise")
# no-change only
usedTactileRotDir      <- c("clockwise", "counterclockwise")
usedMentalRotDir       <- c("none", "clockwise", "counterclockwise")
# only without rotation
usedMentalRotDir       <- c("none")
usedDisparity          <- c("0DEG", "070DEG", "100DEG", "130DEG", "160DEG")
usedLetterIdentities   <- c("R", "L", "F", "P")
# apply filter
data <- subset(data, TactileRotDir %in% usedTactileRotDir)
data <- subset(data, MentalRotDir %in% usedMentalRotDir)

# aggregate data and remove empty cells
data <- aggregate(cbind(RT,ER) ~ ParticipantID*Parity*TactileRotDir*MentalRotDir, data, mean)
data <- droplevels(data);

summary(data)

data$ParticipantID <- factor(data$ParticipantID)

anova_rt <- ezANOVA(data=data, dv = .(RT), wid = .(ParticipantID), within = .(Parity,TactileRotDir), detailed = TRUE)
anova_out(anova_rt)

anova_er <- ezANOVA(data=data, dv = .(ER), wid = .(ParticipantID), within = .(Parity,TactileRotDir), detailed = TRUE)
anova_out(anova_er)