############
#
# This code is licensed under Creative Commons Attribution 4.0 International (CC BY 4.0).
# Your use of this material constitutes acceptance of that license and the conditions of 
# use of materials on the site https://creativecommons.org/licenses/by/4.0/
#
# Copyright 2018, Eberhard Karl University of Tübingen and Juan Purcalla Arrufi.
# Use of these materials permitted only in accordance with license rights granted.
# Materials provided “AS IS”; no representations or
# warranties provided. User assumes all responsibility for use, and all liability related thereto,
# and must independently review all materials for accuracy and efficacy.
#
# ###
#
# This code is a python3.5 implementation of the composition algorithm for the qualitative representation OPRAm, which is
# described in the paper Mossakowski, Moritz, 2012, "Qualitative reasoning about relative direction of oriented points"
# 
# This implementation might be further optimised. 
#
############

#Triangles (given in mossakowski2012qualitative paper) 
triangles = (set(), {(0,0,2), (0,2,0), (1,1,1), (2,0,0), (3,3,3)},
             {(0,0,4), (0,4,0), (1,1,1), (1,1,2), (1,1,3), (1,2,1), (1,3,1),
              (2,1,1), (3,1,1), (4,0,0), (5,7,7), (6,7,7),
              (7,5,7), (7,6,7), (7,7,5), (7,7,6), (7,7,7)},
             {(0,0,6), (0,6,0), (1,1,3), (1,1,4), (1,1,5), (1,2,3), (1,3,1), (1,3,2), (1,3,3), (1,4,1), (1,5,1),
              (2,1,3), (2,2,2), (2,3,1), (3,1,1), (3,1,2), (3,1,3), (3,2,1), (3,3,1),
              (4,1,1), (5,1,1), (6,0,0), (7,11,11), (8,11,11),
              (9,9,11), (9,10,11), (9,11,9), (9,11,10), (9,11,11),
              (10,9,11), (10,10,10), (10,11,9),
              (11,7,11), (11,8,11), (11,9,9), (11,9,10), (11,9,11), (11,10,9), (11,11,7), (11,11,18), (11,11,9)},
             {(0,0,8), (0,8,0), (1,1,5), (1,1,6), (1,1,7), (1,2,5), (1,3,3), (1,3,4), (1,3,5),
              (1,4,3), (1,5,1), (1,5,2), (1,5,3), (1,6,1), (1,7,1),
              (2,1,5), (2,2,4), (2,3,3), (2,4,2), (2,5,1),
              (3,1,3), (3,1,4), (3,1,5), (3,2,3), (3,3,1), (3,3,2), (3,3,3), (3,4,1), (3,5,1),
              (4,1,3), (4,2,2), (4,3,1), (5,1,1), (5,1,2), (5,1,3), (5,2,1), (5,3,1), (6,1,1), (7,1,1),
              (8,0,0), (9,15,15), (10,15,15), (11,13,15), (11,14,15), (11,15,13), (11,15,14), (11,15,15),
              (12,13,15), (12,14,14), (12,15,13), (13,11,15), (13,12,15), (13,13,13), (13,13,14), (13,13,15),
              (13,14,13),(13,15,11), (13,15,12),(13,15,13), (14,11,15), (14,12,14), (14,13,13), (14,14,12),(14,15,11),
              (15,9,15), (15,10,15), (15,11,13), (15,11,14), (15,11,15),(15,12,13), (15,13,11), (15,13,12), (15,13,13),
              (15,14,11), (15,15,9), (15,15,10), (15,15,11)})

#This function normalises the variable according to positive values of
# modulo m, i.e., {0,1,2,...,m-1}
def normalise_mod(i :int, m :int):
    return i % (4*m)

# ===
# These functions check or generate turn triads, as defined in the paper
# ===
def is_turn_values(m, i, j, k):
    AddedValues = normalise_mod(i+j+k, m)
    if AddedValues == 0:
        return True
    elif (AddedValues == 4*m-1 or AddedValues == 1) and (i % 2 == 1) and (j % 2 == 1):
        return True
    else:
        return False

def turn_2values(m, i, j):
    FeasibleTriads = set() #begin with empty set
    for k in range(4*m):
        if is_turn_values(m, i, j, k):
            FeasibleTriads.add((i,j,k)) #Adding the triad to the set
    return FeasibleTriads

def turn_1value(m, i):
    FeasibleTriads = set() #begin with empty set
    for j in range(4*m):
        FeasibleTriads.update(turn_2values(m, i, j)) #Adding triads to the set
    return FeasibleTriads

def turn_0values(m):    
    FeasibleTriads = set() #begin with empty set
    for i in range(4*m):
        FeasibleTriads.update(turn_1value(m, i)) #Adding triads to the set
    return FeasibleTriads

# The function returns the subset of feasible triads that are a turn
# with the given restrictions
#
# In the case no given values verify the triangle it returns the empty set
def turn(m, i=None, j=None, k=None):
    FeasibleTriads = set() #begin with empty set
    
    if k != None:
        if is_turn_values(m, i, j, k):
            FeasibleTriads.add((i,j,k)) #Adding the only checked triad to the set
    elif j != None:
        FeasibleTriads = turn_2values(m, i, j)        
    elif i != None:
        FeasibleTriads = turn_1value(m, i)
    else:
        FeasibleTriads = turn_0values(m)
            
    return FeasibleTriads

# =====================================
# This function returns the composition of two OPRA relations rAB and rBC.
# The OPRA relations are presented as 
# rAB o rBC = rAC; where rAB, rBC, rAC are OPRA relations: 
# - (k,l) for non-superposed
# - (k,) for superposed
# 
# Parameters:
#  * rAB: Qualitative relation between A and B, i.e., A rAB B
#  * rBC: Qualitative relation B and C, i.e., B rBC C
#  * m: Granularity of the representation, i.e., OPRAm
#  * triangles: Valid triangles (provided by the paper)
#
# Return value:
# A tuple with following items:
# 0) The possible s values of the composed relation for
#    binary values (s,t) or for monary values (s,)  
# 1) The possible t values of the composed relation corresponding
#    to binary values (s,t)
# 2) The set of the result compositions, i.e., all (s,t) combinations,
#    and all (s,) monary values.
#    {(s1,t1), (s2,t2), ..., (sN,tN), (s{N+1},), (s{N+2},), ..., (S{M},))
# ========
#
# Example: OPRA1; Composition  (3, 1)  and  (2, 0)  =  ({1, 2, 3}, {1}, {(3, 1), (1, 1), (2, 1)})
# Example: OPRA1; Composition  (1, 3)  o  (2,)  =  set() #Empty set no possible 
# Example: OPRA1; Composition  (1, 2)  and  (2, 0)  =  ({1, 3, (1,)}, {0, 2}, {(1, 2), (3, 0), (1, 0), (3, 2), (1,)}
# Example: OPRA4; Composition  (0, 8)  and  (0, 8)  =  ({0}, {8}, {(0, 8)})
# Example: OPRA4; Composition  (0, 8)  and  (15, 7)  =  ({15}, {8, 9, 7}, {(15, 8), (15, 7), (15, 9)})
#
# ======================================
def compose_opra(rAB,rBC,m,triangles):
    rACSet = set() # The return value: the set of composition results
    tSet = set()
    sSet = set()
    
    #First deciding according to the number of elements in rAB
    if len(rAB) == 2:
        (i, j) = rAB
        if len(rBC) == 1:
            (k,) = rBC
            sSet.add(i) # s = i
            jMinus = normalise_mod(-j, m)
            for t in range(4*m):
                if is_turn_values(m, t, k, jMinus):
                    tSet.add(t)             
        elif len(rBC) == 2:
            (k, l) = rBC
            
            #Computing the non superposed relations (s,t)
            iMinus = normalise_mod(-i, m)
            kMinus = normalise_mod(-k, m)
            
            for (u, v, w) in triangles[m]:
                if is_turn_values(m, v, kMinus, j):
                    for s in range(4*m):
                        if is_turn_values(m, u, iMinus, s):
                            sSet.add(s)
                    for t in range(4*m):
                        tMinus = normalise_mod(-t, m)
                        if is_turn_values(m, w, tMinus, l):
                            tSet.add(t)
                            
            #There are composed relations only if both sets are not empty
            #The composed relations are all combinations of sSet and tSet
            if len(sSet)>0 and len(tSet)>0: 
                for s in sSet:
                    for t in tSet:
                        rACSet.add((s,t))
            
            #Computing the superposed relations (s,)
            if j == k:
                lMinus = normalise_mod(-l, m)
                for s in range(4*m):
                        sMinus = normalise_mod(-s, m)
                        if is_turn_values(m, i, lMinus, sMinus):
                            #The s for superposed relations defer
                            #from those in non-superposed relations
                            sSet.add((s,)) 
                            rACSet.add((s,))
            
    elif len(rAB) == 1:
        (i,) = rAB
        
        if len(rBC) == 2:
            (k, l) = rBC
        elif len(rBC) == 1:
            (k,) = rBC
        else:
            raise ValueError('The relation tuple rBC has neither 1 or 2 elements')
        
        #Calculating the s values of turn(i,k,-s)
        for s in range(4*m):
                sMinus = normalise_mod(-s, m)
                if is_turn_values(m, i, k, sMinus):
                    sSet.add(s)
        
        if len(rBC) == 2:
            tSet.add(l) # t = l
            if len(sSet)>0 and len(tSet)>0: 
                for s in sSet:
                    for t in tSet:
                        rACSet.add((s,t))
        elif len(rBC) == 1:
            (k,) = rBC
            if len(sSet)>0: 
                for s in sSet:
                    rACSet.add((s,))
        else:
            raise ValueError('The relation tuple rBC has neither 1 or 2 elements')
    else: # elements in rAB
        raise ValueError('The relation tuple rAB has neither 1 or 2 elements')
                
    return (sSet, tSet, rACSet)