/**
 * SimulationManager header file. Should be included by plugins that use SimulationManager-functions (Header)
 * 
 * Copyright 2013 Fabian Schrodt, FSchrodt@gmx.de
 * 
 * This file is part of SimulationManager.
 * 
 * SimulationManager is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License version 3 as published by the Free Software Foundation.
 * 
 * SimulationManager is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with SimulationManager. If not, see http://www.gnu.org/licenses/.
 */

#pragma once

#include "SimulationCommand.h"
#include "Simulation.h"
#include "PipeServerSimulation.h"
#include "PluginInterface.h"

#include <vector>
#include <map>
#include <deque>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

struct cmp_plug
{
   bool operator()(PluginInterface* a, PluginInterface* b)
   {
      return a < b;
   }
};

class SimulationManager
{
	pthread_mutex_t* man_lock;

	vector <PipeServerSimulation*> simulations;

	unsigned int activeSimulations;

	map <PluginInterface*, void*, cmp_plug> plugins;	//Plugin-Object, Handle, Comparator

	bool validID(unsigned int simID);

	/*
		Adds a new coded Simulation to the Manager
		Returns simulation-ID if successfull
		Returns -1 if full
		Returns -2 if unknown type
	*/
	int addSimulation(SimulationInfo* set_simInfo);

	/*
	 * may be called only by addSimulation!
	 * TODO: Check if Process was closed correctly!
	 * */
	void deleteSimulation(unsigned int simID);

	/*
	 * Loads shared objects located in plugins.txt as plugins that implement the PluginInterface
	 * */
	void loadPlugins();

	/*
	 * Internal function that is used to block the caller of a StopSimulationNow-command until the simulation has finished.
	 * Returns the number of ms remaining.
	 * */
	unsigned int waitForSimulationToFinish(unsigned int simID, unsigned int max_useconds);


	/*
	 * All public functions must lock the manager for thread safety.
	 * */
public:

	/*
	 * Manually load plugin at runtime
	 * Returns a pointer to the plugin on success or NULL on failure.
	 * */
	PluginInterface* loadPlugin(const char* soname);

	/*
	 * Forces unloading of plugin.
	 * */
	int unloadPlugin(PluginInterface* plugin);

	/*
	 * Each sim may have its own sensorHandler using this function.
	 * Returns 0 on success.
	 */
	int setSensorHandler(SensorHandlerInterface* set_sensorHandler, unsigned int simID);

	/*
	 * Default function to execute commands.
	 * The execution of incoming (non-blocking) commands will be parallelized according to the simulation they are assigned to.
	 * */
	SimulationCommand applyCommand(SimulationCommand* cmd);

	/*
	 * Default konstructor
	 * */
	SimulationManager();

	/*
	 * Default destructor
	 * */
	~SimulationManager();


	// now some statistical information:
	struct Statistics
	{
		// get software version as String
		const char* version();

		// get startup date/time of the server as String
		const char* onlineSince();

		// get total number of simulation slots
		int numSlots();

		// get number of currently busy slots
		int numBusySlots();

		// get amount of currently free memory (in MB)
		double getMemFreeMB();

		// get current CPU load
		double currentLoad();

		// total amount of non-idle cpu time in seconds used on all cores since startup
		double totalCpuTimeSeconds();

		// total number rejected slot requests
		int numSlotRequestsRejected();

		// total number of simulations done (i.e. not including the currently running)
		int numFinishedSimulations();

		// total number of simulation steps so far in all simulations (done & active)
		int numStepsInFinishedSimulations();

		// -------------------

		// get total cumulative time since startup for
		// which the specified number of simulations were active simultaneously)
		double activeNumSlotsTimeSeconds(int numBusySlots);

		// get total average of CPU load (averaged over all times since startup at
		// which the specified number of simulations were active simultaneously)
		double averageLoad(int numBusySlots);

		// -------------------

		// name of simulation in the slot specified, null if the slot is not busy
		const char* simulationName(int slotNumber);

		// number of simulation steps done so far in the simulation in the slot specified
		int numStepsSoFar(int slotNumber);

		// the average time in ms for processing a time step in the simulation in the slot specified
		double averageStepTimeMiliSeconds(int slotNumber);

		Statistics(SimulationManager* set_man)
		{
			man=set_man;
		}

	private:
		SimulationManager* man;
	} stats;
};
