/**
 * Tools for creating Neural Networks (Header+Implementation)
 * 
 * This file is licensed under GPL.
 * 
 * Copyright 2013 Fabian Schrodt, FSchrodt@gmx.de
 * 
 * This file is part of RNNPBlib.
 * 
 * RNNPBlib is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License version 3 as published by the Free Software Foundation.
 * 
 * RNNPBlib is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with RNNPBlib. If not, see http://www.gnu.org/licenses/.
 */

#pragma once

#include "RNNPB_Definitions.h"
#include "RNNPB_NeuronLayer.h"
#include "RNNPB_Weight.h"

struct RNNPB_AssemblyTools
{
	static void connectLayersSecondorderCV(RNNPB_NeuronLayer* from_layer, RNNPB_NeuronLayer* to_layer, vector <RNNPB_NeuronLayer*>* cl_container)
	{
		RNNPB_NeuronLayer* weight_layer = new RNNPB_NeuronLayer(from_layer->size()*to_layer->size(), Linear);

		int pb_count=0;
		for(unsigned int i=0;i<from_layer->size();i++)
		{
			for(unsigned int j=0;j<to_layer->size();j++)
			{			
				new RNNPB_SecondOrderWeightCV(from_layer->neuron[i], to_layer->neuron[j], weight_layer->neuron[pb_count]);
				pb_count++;
			}
		}

		cl_container->push_back(weight_layer);
	}

	static void connectLayersModulatedCV(RNNPB_NeuronLayer* from_layer, RNNPB_NeuronLayer* to_layer, vector <RNNPB_NeuronLayer*>* cl_container)
	{
		RNNPB_NeuronLayer* weight_layer = new RNNPB_NeuronLayer(from_layer->size()*to_layer->size(), Linear);

		int pb_count=0;
		for(unsigned int i=0;i<from_layer->size();i++)
		{
			for(unsigned int j=0;j<to_layer->size();j++)
			{			
				new RNNPB_ModulatedWeightCV(from_layer->neuron[i], to_layer->neuron[j], weight_layer->neuron[pb_count]);
				pb_count++;
			}
		}

		cl_container->push_back(weight_layer);
	}

	static void connectLayersSecondorder(RNNPB_NeuronLayer* from_layer, RNNPB_NeuronLayer* to_layer, RNNPB_NeuronLayer* injection)
	{
		#ifdef ENABLE_DEBUG
				if(injection->size()!=from_layer->size()*to_layer->size())
					cout<<"Error: Wrong injection layer size!\n";
		#endif

		int pb_count=0;
		for(unsigned int i=0;i<from_layer->size();i++)
		{
			for(unsigned int j=0;j<to_layer->size();j++)
			{
				new RNNPB_SecondOrderWeight(from_layer->neuron[i], to_layer->neuron[j], injection->neuron[pb_count]);
				pb_count++;
			}
		}
	}

	static void connectLayersModulated(RNNPB_NeuronLayer* from_layer, RNNPB_NeuronLayer* to_layer, RNNPB_NeuronLayer* injection)
	{
		#ifdef ENABLE_DEBUG
				if(injection->size()!=from_layer->size()*to_layer->size())
					cout<<"Error: Wrong injection layer size!\n";
		#endif

		int pb_count=0;
		for(unsigned int i=0;i<from_layer->size();i++)
		{
			for(unsigned int j=0;j<to_layer->size();j++)
			{
				new RNNPB_ModulatedWeight(from_layer->neuron[i], to_layer->neuron[j], injection->neuron[pb_count]);
				pb_count++;
			}
		}
	}

	static void connectLayers(RNNPB_NeuronLayer* from_layer, RNNPB_NeuronLayer* to_layer, double set_learning_rate = 0)
	{
		for(unsigned int i=0;i<from_layer->size();i++)
		{
			for(unsigned int j=0;j<to_layer->size();j++)
			{
				RNNPB_NormalWeight* tmp = new RNNPB_NormalWeight(from_layer->neuron[i], to_layer->neuron[j]);
				if(set_learning_rate!=0)
					tmp->learningRate=set_learning_rate;
			}
		}
	}

	static void connectLayersConstant(RNNPB_NeuronLayer* from_layer, RNNPB_NeuronLayer* to_layer, double set_weight = 0)
	{
		for(unsigned int i=0;i<from_layer->size();i++)
		{
			for(unsigned int j=0;j<to_layer->size();j++)
			{
				RNNPB_NormalWeight* tmp = new RNNPB_NormalWeight(from_layer->neuron[i], to_layer->neuron[j]);
				if(set_weight!=0)
					tmp->weight=set_weight;
				tmp->constant=true;
			}
		}
	}

	static void connectLayersDelayed(RNNPB_NeuronLayer* from_layer, RNNPB_NeuronLayer* to_layer, unsigned int delaySteps)
	{
		for(unsigned int i=0;i<from_layer->size();i++)
		{
			for(unsigned int j=0;j<to_layer->size();j++)
			{
				new RNNPB_TapDelayLineWeight(from_layer->neuron[i], to_layer->neuron[j], delaySteps);
			}
		}
	}

	static void connectLayersDelayedOneToOne(RNNPB_NeuronLayer* from_layer, RNNPB_NeuronLayer* to_layer, unsigned int delaySteps)
	{
		#ifdef ENABLE_DEBUG
			if(from_layer->size()!=to_layer->size())
				cout<<"Error: Wrong layer size!\n";
		#endif

		for(unsigned int i=0;i<from_layer->size();i++)
		{
			new RNNPB_TapDelayLineWeight(from_layer->neuron[i], to_layer->neuron[i], delaySteps);
		}
	}

	static void addParametricBiasToLayer(RNNPB_NeuronLayer* to_layer, vector <RNNPB_NeuronLayer*>* cl_container)
	{
		RNNPB_NeuronLayer* from_layer = new RNNPB_NeuronLayer(to_layer->size(), Linear, 0.0, -1.0);

		for(unsigned int j=0;j<to_layer->size();j++)
		{
			new RNNPB_ParametricBiasWeightCV(from_layer->neuron[j], to_layer->neuron[j]);
		}

		cl_container->push_back(from_layer);
	}

	static void addBiasToLayer(RNNPB_NeuronLayer* to_layer)
	{
		RNNPB_NeuronLayer* from_layer = new RNNPB_NeuronLayer(1, Linear, 0.0, -1.0);

		for(unsigned int j=0;j<to_layer->size();j++)
		{			
			new RNNPB_NormalWeight(from_layer->neuron[0], to_layer->neuron[j]);
		}
	}

	static void addRecurrency(RNNPB_NeuronLayer* from, unsigned int neuron_nr_from, RNNPB_NeuronLayer* to, unsigned int neuron_nr_to, unsigned int delaySteps)
	{
		new RNNPB_TapDelayLineWeight(from->neuron[neuron_nr_from], to->neuron[neuron_nr_to], delaySteps);
	}

	static void connectReservoir(RNNPB_NeuronLayer* layer, unsigned int delaySteps = 1)
	{
		for(unsigned int i=0;i<layer->size();i++)
		{
			for(unsigned int j=0;j<layer->size();j++)
			{
				if(i!=j)
				{
					new RNNPB_TapDelayLineWeight(layer->neuron[i], layer->neuron[j], delaySteps);
					
					//CHECK: Intralayer-Konkurrenz!
					//tmp->weight=DEFAULT_INTRALAYER_WEIGHT;
				}
			}
		}
	}

	static void addSelfInhibitionToLayer(RNNPB_NeuronLayer* layer, unsigned int delaySteps = 1)
	{
		for(unsigned int i=0;i<layer->size();i++)
		{
			new RNNPB_TapDelayLineWeight(layer->neuron[i], layer->neuron[i], delaySteps);
		}
	}
};
