/**
 * Neurons (Implementation)
 * 
 * Copyright 2013 Fabian Schrodt, FSchrodt@gmx.de
 * 
 * This file is part of RNNPBlib.
 * 
 * RNNPBlib is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License version 3 as published by the Free Software Foundation.
 * 
 * RNNPBlib is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with RNNPBlib. If not, see http://www.gnu.org/licenses/.
 */

#include "RNNPB_Neuron.h"

double RNNPB_Neuron::getFeedback()
{
	#ifdef ENABLE_DEBUG
	if(!feedbackUptodate)
	{
		//cout<<"Warning at iteration "<<iteration<<": Using delay-weights or strange network architecture. Or something is going wrong!\n";
		//__asm{int 3};
	}
	if(!isOutputNeuron)
	{
		cout<<"Error: Why do you ask me?!\n";
		//__asm{int 3};
	}
	#endif

	return feedback;
}

double RNNPB_Neuron::getInput()
{
	#ifdef ENABLE_DEBUG
	if(constant && !inputUptodate)
		cout<<"Error: No input value set for constant neuron!\n";
	#endif

	if(inputUptodate || constant)
		return input;

	for(unsigned int i=0;i<incoming_weights.size();i++)
		input += incoming_weights[i]->getOutput();

	inputUptodate=true;

	return input * activationSteepness;
}

double RNNPB_Neuron::getRadialInput()
{
	RNNPB_Vector weights(incoming_weights.size());
	RNNPB_Vector activations(incoming_weights.size());
	for(unsigned int i=0;i<weights.size;i++)
	{
		weights[i]=incoming_weights[i]->getWeight();
		activations[i]=incoming_weights[i]->getOutput()/incoming_weights[i]->getWeight(); //TODO: implement getPredecessorActivation() for delayed weights!
	}

	/*cout<<"CENTER IS ";
	weights.print();
	cout<<endl;*/

	return RNNPB_Vector::euclideanDistance(&weights, &activations) * activationSteepness;		//Normal implementierung. In the strict sense this does not allow the derivation of the backpropagation algorithm (but RNNPB_Vector::halfSquaredError(...) may deform the input space!)
}

void RNNPB_Neuron::iterate()
{
	iteration++;

	deltaErrorUptodate = false;
	feedbackUptodate = false;

	*deltaError=0.0;

	if(!constant)
	{
		input=0.0;
		inputUptodate = false;
		activationUptodate = false;
		activationDifferentialUptodate = false;
	}
}

void RNNPB_Neuron::clear()
{
	deltaErrorUptodate = false;
	feedbackUptodate = false;
	activationUptodate = false;
	activationDifferentialUptodate = false;

	iteration=0;
	*deltaError=0.0;
	feedback=0.0;
	*activation=0.0;
	*activationDifferential=0.0;

	if(!constant)
	{
		input=0.0;
		inputUptodate = false;
	}
}

void RNNPB_Neuron::subscribeNetwork(RNNPB_NetworkContainer* net, int run)
{
	if(parsed==run)
		return;

	parsed=run;
	network=net;

	if(run==1)	//forward
	{
		for(unsigned int n=0;n<incoming_weights.size();n++)
		{
			incoming_weights[n]->subscribeNetwork(net, run);
		}
	}
	else		//backward
	{
		for(unsigned int n=0;n<outgoing_weights.size();n++)
		{
			outgoing_weights[n]->subscribeNetwork(net, run);
		}
	}

	network->registerNeuron(this, run);
}

RNNPB_NeuronLayer* RNNPB_Neuron::getNeuronLayer()
{
	return neuron_layer;
}

double RNNPB_Neuron::getDeltaError()
{
	if(deltaErrorUptodate)
		return *deltaError;

	if(isOutputNeuron)
	{
		*deltaError = getActivationDifferential() * getFeedback();
	}
	else	//for hidden / input / PB neurons
	{
		//*deltaError = 0.0;	//done by iterate()

		for(unsigned int i=0;i<outgoing_weights.size();i++)
		{
			*deltaError += outgoing_weights[i]->getWeightedDeltaError(this);
		}

		*deltaError *= getActivationDifferential();
	}

	deltaErrorUptodate = true;
	return *deltaError;
}

double RNNPB_Neuron::getActivationDifferential()  //TODO: Use constant signum for non-radial neurons!
{
	if(activationDifferentialUptodate)
		return *activationDifferential;

	switch(activationType)
	{
		case Logistic:
			//lookup:
			#ifdef ENABLE_LOOKUP_TABLES
			*activationDifferential=getInput();
			if(*activationDifferential>=LUT_SPAN || *activationDifferential<-LUT_SPAN)
			{
				*activationDifferential
					= activationSteepness / (4.0 * pow(cosh(*activationDifferential / 2.0),2.0));
			}
			else
			{
				*activationDifferential
					= network->lut.sigmoidDiffLUT[int(*activationDifferential*(LUT_SIZE/(2.0*LUT_SPAN))+(LUT_SIZE/2.0))] * activationSteepness;
			}
			#else
			//calc:
			*activationDifferential
					= activationSteepness / (4.0 * pow(cosh(getInput() / 2.0),2.0));
			#endif
			break;
		case Tanh:
			*activationDifferential=getInput();
			#ifdef ENABLE_LOOKUP_TABLES
			if(*activationDifferential>=LUT_SPAN || *activationDifferential<-LUT_SPAN)
			{
				*activationDifferential
					= activationSteepness / pow(cosh(*activationDifferential),2.0);
			}
			else
			{
				*activationDifferential
					= network->lut.tanhDiffLUT[int(*activationDifferential*(LUT_SIZE/(2.0*LUT_SPAN))+(LUT_SIZE/2.0))] * activationSteepness;

				//cout<<"Lookup differential deviation: "<<4.0 * (pow(cosh(getInput()),2.0)) / pow(cosh(2.0*getInput()) + 1.0, 2.0) - *activationDifferential<<endl;
			}
			#else
			//calc:	//TODO: dont use cosh, use exp oder EXP!!!
			*activationDifferential
				= activationSteepness / pow(cosh(*activationDifferential),2.0);
			#endif
			break;
		case QuasiLinear:
			*activationDifferential=getInput();
			if(*activationDifferential>1.0 || *activationDifferential<0.0)
				*activationDifferential = 0.0;
			else
				*activationDifferential = activationSteepness;
			break;
		case Linear:
			*activationDifferential = activationSteepness;
			break;
		case Heaviside:
			*activationDifferential = 0.0;
			break;
		case RadialLinear:
			*activationDifferential=getRadialInput();
			if(*activationDifferential > 1.0/activationSteepness)
				*activationDifferential = -0.0000001;	//keep signum
			else if(*activationDifferential < - 1.0/activationSteepness)
				*activationDifferential = 0.0000001;	//keep signum
			else if(*activationDifferential >= 0)
				*activationDifferential = - activationSteepness;
			else
				*activationDifferential = activationSteepness;
			break;
		case RadialGaussian:	// normalized Gaussian function
			*activationDifferential=getRadialInput();
			#ifdef ENABLE_LOOKUP_TABLES
			if(*activationDifferential>=LUT_SPAN || *activationDifferential<-LUT_SPAN)
			{
				*activationDifferential
					= exp(-0.5*pow(*activationDifferential,2.0))*(-(*activationDifferential)*activationSteepness);
			}
			else
			{
				*activationDifferential
					= network->lut.gaussDiffLUT[int(*activationDifferential*(LUT_SIZE/(2.0*LUT_SPAN))+(LUT_SIZE/2.0))] * activationSteepness;
			}
			#else
			//calc:
			*activationDifferential
				= exp(-0.5*pow(*activationDifferential,2.0))*(-(*activationDifferential)*activationSteepness);
			#endif
			break;
		#ifdef ENABLE_DEBUG
		default:
			cout<<"Activation error!\n";
			*activationDifferential = 0;
			break;
		#endif
	}

	#ifdef ENABLE_FLAT_SPOT_ELIM
	double minimum = std::abs(MINIMUM_DIFFERENTIAL);
	if(std::abs(*activationDifferential) < minimum)
	{
		*activationDifferential = RNNPB_Helper::signum(*activationDifferential) * minimum;
	}
	#endif

	//cout<<*activationDifferential<<endl;
	activationDifferentialUptodate = true;
	return *activationDifferential;
}

double RNNPB_Neuron::getInverseActivationOf(double output)
{
	switch(activationType)
	{
		case QuasiLinear:
			if(output >= 1.0 || output <= 0.0)
			{
				cout<<"Error: QuasiLinear^-1 of "<<output<<" is undefined!\n";
				//__asm{int 3};
				return 0;
			}
			return output;
		case Linear:
			return output;
		case Logistic:
			#ifdef ENABLE_DEBUG
			if(output >= 1.0 || output <= 0.0)
			{
				cout<<"Error: Logistic^-1 of "<<output<<" is undefined!\n";
				//__asm{int 3};
				return 0;
			}
			#endif
			return log(-output / (output - 1.0)) / activationSteepness;
		case Tanh:
			#ifdef ENABLE_DEBUG
			if(output >= 1.0 || output <=-1.0)
			{
				cout<<"Error: Tanh^-1 of "<<output<<" is undefined!\n";
				//__asm{int 3};
				return 0;
			}
			#endif
			//return atanh(output);	//doesnt work for ms visual stdio
			return (log(1.0+output) - log(1.0-output))/2.0;
		case RadialLinear:
		case Heaviside:
		case RadialGaussian:
			#ifdef ENABLE_DEBUG
			if(output >= 1.0f || output <= 0.0)
			{
				cout<<"Error: Ambiguous inverse!\n";
				//__asm{int 3};
			}
			#endif
			return 0;
		#ifdef ENABLE_DEBUG
		default:
			cout<<"Activation error!\n";
			return 0;
		#endif
	}
	return 0;
}

double RNNPB_Neuron::getActivation()
{
	if(activationUptodate)
		return *activation;

	#ifdef ENABLE_DEBUG
	*activation=getInput();
	if(*activation!=*activation)
	{
		cout<<"Warning: Activation NaN!\n";
		return 0.0;
	}
	#endif

	switch(activationType)
	{
		case Logistic:
			//lookup:
			#ifdef ENABLE_LOOKUP_TABLES
			*activation = getInput();
			if(*activation >= LUT_SPAN || *activation < -LUT_SPAN)
				#ifdef ENABLE_ENABLE_EXP_APPROX
					*activation = 1.0 / (1.0 + EXP(- *activation));
				#else
					*activation = 1.0 / (1.0 + exp(- *activation));
				#endif
			else
				*activation = network->lut.sigmoidLUT[int(*activation*(LUT_SIZE/(2.0*LUT_SPAN))+(LUT_SIZE/2.0))];
			#else
			//calc:
			#ifdef ENABLE_ENABLE_EXP_APPROX
				*activation = 1.0 / (1.0 + EXP(- getInput()));
			#else
				*activation = 1.0 / (1.0 + exp(- getInput()));
			#endif
			#endif
			break;
		case Tanh:
			//lookup:
			#ifdef ENABLE_LOOKUP_TABLES
			*activation = getInput();
			if(*activation >= LUT_SPAN || *activation < -LUT_SPAN)
				*activation = tanh(*activation);
			else
			{
				*activation = network->lut.tanhLUT[int(*activation*(LUT_SIZE/(2.0*LUT_SPAN))+(LUT_SIZE/2.0))];

				//cout<<"Lookup deviation: "<<tanh(getInput()) - *activation<<endl;
			}
			#else
				//calc:
				//*activation = tanh(getInput());	//slow
				#ifdef EXP_APPROX
				*activation = 1.0 - 2.0 / (EXP(2.0*getInput()) + 1.0);
				#else
				*activation = 1.0 - 2.0 / (exp(2.0*getInput()) + 1.0);
				#endif
			#endif
			break;
		case QuasiLinear:
			*activation = getInput();
			if(*activation>1.0)
				*activation=1.0;
			else if(*activation<0.0)
				*activation=0.0;
			break;
		case Linear:
			*activation = getInput();
			break;
		case Heaviside:
			if(getInput()>=0.5) *activation = 1.0;
			else *activation = 0.0;
			break;
		case RadialGaussian:	//normalized Gaussian function
			//lookup:
			#ifdef ENABLE_LOOKUP_TABLES
			*activation = getRadialInput();
			if(*activation >= LUT_SPAN || *activation < -LUT_SPAN)
				*activation = exp(-0.5*pow(*activation,2.0));

			else
			{
				*activation = network->lut.gaussLUT[int(*activation*(LUT_SIZE/(2.0*LUT_SPAN))+(LUT_SIZE/2.0))];

				//cout<<"Lookup deviation: "<<tanh(getInput()) - *activation<<endl;
			}
			#else
			//calc:
			*activation = exp(-0.5*pow(getRadialInput(),2.0));
			#endif
			break;
		case RadialLinear:	//Linear radial Function (sawtooth function)
			*activation = max(1.0 - getRadialInput(), 0.0);		//abs not required here, since raidal input is > 0
			break;
		#ifdef ENABLE_DEBUG
		default:
			cout<<"Activation error!\n";
			*activation = 0;
			break;
		#endif
	}

	activationUptodate=true;
	return *activation;
}


//Error
void RNNPB_Neuron::setFeedback(double set_feedback)
{
	feedback = set_feedback;
	feedbackUptodate=true;
}

//"FeedForward"
void RNNPB_Neuron::setInput(double set_input)
{
	#ifdef ENABLE_DEBUG
	if(constant && iteration>1)
		cout<<"Warning: Set constant neuron input at iteration "<<iteration<<"\n";
	/*else if(constant)
	{
		cout<<"Set constant neuron initially to "<<set_input<<". Thats OK if its parametric."<<endl;
		usleep(10000);
	}*/
	#endif

	input = set_input;

	inputUptodate=true;
	activationUptodate = false;
	activationDifferentialUptodate = false;
}

RNNPB_Neuron::RNNPB_Neuron(RNNPB_NeuronLayer* set_neuron_layer, ActivationType set_type = Linear, double set_activation_parameter = 0.0, double set_constant_activation = 0.0)
{
	network=NULL;

	neuron_layer=set_neuron_layer;

	//clear(); //clear after allocating memory!

	constant=false;
	isRadial=false;

	switch(set_type)
	{
	case Logistic:
		activationSteepness=DEFAULT_SIGMOID_STEEPNESS;
		break;
	case Tanh:
		activationSteepness=DEFAULT_TANH_STEEPNESS;
		break;
	case RadialGaussian:
		activationSteepness=DEFAULT_GAUSSIAN_STEEPNESS;
		isRadial=true;
		break;
	case RadialLinear:
		activationSteepness=DEFAULT_RADIAL_LINEAR_STEEPNESS;
		isRadial=true;
		break;
	default:
		activationSteepness=DEFAULT_STEEPNESS;
		break;
	}

	if(set_activation_parameter != 0.0)
		activationSteepness=set_activation_parameter;

	isOutputNeuron=false;

	activationType = set_type;

	if(set_constant_activation!=0.0)
	{
		constant = true;
		input = set_constant_activation;
		inputUptodate = true;
	}

	//self-contained allocation needed for autoadaptive network topologies
	activation=new double;
	activationDifferential=new double;
	deltaError=new double; //especially necessary for bias neurons with inverse buffering

	parsed=0;
}

RNNPB_Neuron::~RNNPB_Neuron()
{
}
