/**
 * Layer of Neurons (Implementation)
 * 
 * Copyright 2013 Fabian Schrodt, FSchrodt@gmx.de
 * 
 * This file is part of RNNPBlib.
 * 
 * RNNPBlib is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License version 3 as published by the Free Software Foundation.
 * 
 * RNNPBlib is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with RNNPBlib. If not, see http://www.gnu.org/licenses/.
 */

#include <pthread.h>

#include "RNNPB_NeuronLayer.h"

/*
 * Concept-vectors will be updated by transfering the updated subsequent weight to the concept elements.
 * This way the weights can be learned by any appropriate learning algorithm (i.e. backpropagation).
 * This will be done after an epoch, the bias-weights will be reset after that.
*/
RNNPB_Vector RNNPB_NeuronLayer::swapWeights()
{
	RNNPB_Vector ret(neuron.size());

	for(unsigned int i=0;i<neuron.size();i++)
	{
		switch(neuron[i]->outgoing_weights[0]->weightType)
		{
		case SecondOrderCV:
			ret[i] = neuron[i]->getInverseActivationOf(  neuron[i]->outgoing_weights[0]->weight  );
			//neuron[i]->outgoing_weights[0]->weight = 1.0; //Nicht nötig!
			break;
		case ModulatedCV:
			ret[i] = neuron[i]->getInverseActivationOf(  ((RNNPB_ModulatedWeightCV*) neuron[i]->outgoing_weights[0])->pb_weight  );
			break;
		case ParametricBiasCV:
			//cout<<"neuron[i]->getActivation()="<<neuron[i]->getActivation()<<endl;
			ret[i] = neuron[i]->getInverseActivationOf(  neuron[i]->getActivation() * neuron[i]->outgoing_weights[0]->weight );
			neuron[i]->outgoing_weights[0]->weight = 1.0;
			break;
		default:
			cout<<"Inappropriate weight type!\n";
		}		
	}

	/*cout<<"GOT CONCEPT VECTOR: ";
	ret.print();
	cout<<endl;*/

	return ret;
}

unsigned int RNNPB_NeuronLayer::size()
{
	return neuron.size();
}

void RNNPB_NeuronLayer::setConstant(bool set)
{
	for(unsigned int i=0;i<neuron.size();i++)
		neuron[i]->constant=set;
}

void RNNPB_NeuronLayer::setOutgoingWeightsConstant(bool set)
{
	for(unsigned int i=0;i<neuron.size();i++)
	{
		for(unsigned int j=0;j<neuron[i]->outgoing_weights.size();j++)
			neuron[i]->outgoing_weights[j]->constant=set;
	}
}

RNNPB_Vector RNNPB_NeuronLayer::getActivation()
{
	RNNPB_Vector tmp(neuron.size());

	for(unsigned int i=0;i<neuron.size();i++)
	{
		tmp[i]=neuron[i]->getActivation();
	}

	return tmp;
}

RNNPB_Vector RNNPB_NeuronLayer::getInput()
{
	RNNPB_Vector tmp(neuron.size());

	for(unsigned int i=0;i<neuron.size();i++)
	{
		tmp[i]=neuron[i]->getInput();
	}

	return tmp;
}

void RNNPB_NeuronLayer::setInput(RNNPB_Vector set_input)
{
	for(unsigned int i=0;i<neuron.size();i++)
	{
		neuron[i]->setInput(set_input[i]);
	}
}

void RNNPB_NeuronLayer::setFeedback(RNNPB_Vector set_input)
{
	#ifdef ENABLE_DEBUG
	if(set_input.size!=neuron.size())
	{
		cout<<"Dimension error: Layer size = "<<neuron.size()<<", Input size = "<<set_input.size<<endl;
		//__asm{int 3};
	}
	#endif

	for(unsigned int i=0;i<neuron.size();i++)
	{
		neuron[i]->setFeedback(set_input[i]);
	}
}

void RNNPB_NeuronLayer::parseNetwork(RNNPB_NetworkContainer* parent, int run)
{
	if(parsed==run)
		return;
	parsed=run;

	for(unsigned int i=0;i<neuron.size();i++)
	{
		if(run==1)
			neuron[i]->isOutputNeuron=true;

		neuron[i]->subscribeNetwork(parent, run);
	}
}

RNNPB_NeuronLayer::RNNPB_NeuronLayer(unsigned int size, ActivationType set_type, double set_activation_parameter, double set_constant_activation)
{
	#ifdef ENABLE_DEBUG
	cout<<"Creating NeuronLayer of size "<<size<<", type "<<set_type<<", constant "<<set_constant_activation<<endl;
	if(set_constant_activation != 0.0 && set_type!=Linear)
		cout<<"Warning: Constant neuron should have linear activation function!\n";
	#endif

	parsed=0;

	neuron.resize(size, NULL);

	for(unsigned int i=0;i<size;i++)
	{
		neuron[i] = new RNNPB_Neuron(this, set_type, set_activation_parameter, set_constant_activation);
	}
}
