/**
 * Iterface for a SimulationManager controlled Simulation (as a pipe client receiving simulation commands) (Header+Implementation)
 * 
 * Copyright 2013 Fabian Schrodt, FSchrodt@gmx.de
 * 
 * This file is part of SimulationManager.
 * 
 * SimulationManager is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License version 3 as published by the Free Software Foundation.
 * 
 * SimulationManager is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with SimulationManager. If not, see http://www.gnu.org/licenses/.
 */

#pragma once

#include "Simulation.h"
#include "PipeClient.h"
#include "SimulationCommand.h"

#include <iostream>
#include <unistd.h>

using namespace std;

class PipeClientSimulation : public Simulation, public PipeClient
{
	StreamableDataContainer* dat;

	virtual void setAction(const StreamableDataContainer* set_data) = 0;

	virtual StreamableDataContainer* getSensors() = 0;

	virtual void step(unsigned int nr_of_steps) = 0;

	void applyCommand()
	{
		//cout<<"TEST: "<<*((MessageType*) readbuffer)<<endl;

		switch(*((MessageType*) readbuffer))
		{
		case Step:
		{
			readDataToBuffer();
			unsigned int nr_of_steps = *((unsigned int*) readbuffer);
			step(nr_of_steps);
			steps+=nr_of_steps;
			break;
		}
		case SetAction:
		{
			unsigned int bytesize = readDataToBuffer();

			if(dat==NULL)
				dat=new StreamableDataContainer(bytesize);
			dat->setDataStream(readbuffer,bytesize);

			setAction(dat);

			break;
		}
		case GetSensors:
		{
			MessageType mess=ACKGetSensors;
			writeData((unsigned char*) &mess, sizeof(mess));

			StreamableDataContainer* answer = getSensors();

			writeData((unsigned char*) answer->getDataStream(), answer->getByteSize());

			delete answer;

			flushWriteBuffer();
			break;
		}
		case StartSimulation:	//send PID (and stuff?) to manager
		{
			MessageType mess=StartSimulation;
			writeData((unsigned char*) &mess, sizeof(mess));
			pid_t myPID = getpid();
			writeData((unsigned char*) &myPID, sizeof(myPID));
			break;
		}
		case StopSimulation:
		{
			stopSimulation();
			break;
		}
		default:
		{
			cout<<"SIMULATION ERROR: UNKNOWN COMMAND '"<<readbuffer<<"'\n";
			break;
		}
		}
	}

protected:
	PipeClientSimulation(unsigned int argc, char** argv, unsigned int set_buffersize, char* output_pipename, char* input_pipename) : Simulation(argc, argv), PipeClient(set_buffersize, output_pipename, input_pipename)
	{
		dat=NULL;
	}

	~PipeClientSimulation()
	{
		//indicate end of simulation to server
		MessageType mess=StopSimulation;
		writeData((unsigned char*) &mess, sizeof(mess));

		//close everything
		fclose(input_fifo);
		fclose(output_fifo);
	}

public:
	//CHECK: Kann selbstständiges Beenden der Sim die GetSensorsQueue auf Manager-Seite verwirren??
	void stopSimulation()
	{
		//indicate end of simulation to server
		MessageType mess=StopSimulation;
		writeData((unsigned char*) &mess, sizeof(mess));

		//close everything
		fclose(input_fifo);
		fclose(output_fifo);
		exit(0);
	}
};
