/**
 * Manager-sided control structure for simulation representation (Header+Implementation)
 * 
 * Copyright 2013 Fabian Schrodt, FSchrodt@gmx.de
 * 
 * This file is part of SimulationManager.
 * 
 * SimulationManager is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License version 3 as published by the Free Software Foundation.
 * 
 * SimulationManager is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with SimulationManager. If not, see http://www.gnu.org/licenses/.
 */

#pragma once

#include <iostream>
#include <pthread.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "SensorHandlerInterface.h"

#include "PipeRW.h"

using namespace std;

class PipeServerSimulation : public Simulation, public PipeRW
{
	char* output_pipename;
	char* input_pipename;

	unsigned int simID;

	pthread_mutex_t* pidLock;
		pid_t PID; //is set delayed!! initially 0

	char* cmdline;

	pthread_mutex_t* queueLock;
		deque <SimulationCommand> commandQueue;
	pthread_mutex_t* emptyQueueLock;

	SensorHandlerInterface* sensorHandler;

protected:
	pthread_mutex_t* sensorLock;
		deque <SimulationCommand> queuedGetSensorCommands;
		pthread_mutex_t* sensorsDemanded;
		StreamableDataContainer* gotSensorsNow;

	pthread_t queueThread;
	pthread_t sensorThread;

	bool shutDown;	//indicates if the simulation will be shut down soon
	pthread_mutex_t* deleteMeLock; //indicated if the simulation can be deleted (that is, sensorThread has finished)


private:

	void setAction(const StreamableDataContainer* set_data)
	{
		MessageType mess=SetAction;
		writeData((unsigned char*) &mess, sizeof(mess));
		writeData((unsigned char*) set_data->getDataStream(), set_data->getByteSize());
		flushWriteBuffer();
	}

	StreamableDataContainer* getSensors()
	{
		MessageType mess=GetSensors;
		writeData((unsigned char*) &mess, sizeof(mess));
		flushWriteBuffer();

		//data will be returned somewhen else. return value comes from deriving from Simulation where it is useful on simulation-side
		return NULL;
	}

	void step(unsigned int nr_of_steps)
	{
		MessageType mess=Step;
		writeData((unsigned char*) &mess, sizeof(mess));
		writeData((unsigned char*) &nr_of_steps, sizeof(nr_of_steps));
		flushWriteBuffer();
		steps++;
	}

	void runSensorReader()
	{
		while(1)
		{			
			readDataToBuffer();
			MessageType mess = *((MessageType*) readbuffer);

			switch(mess)
			{
			case ACKGetSensors:
			{
				unsigned int bytesize = readDataToBuffer();

				pthread_mutex_lock(sensorLock);

				if(queuedGetSensorCommands.size()==0)	//Command was GetSensorsNow
				{
					pthread_mutex_unlock(sensorLock);

					if(gotSensorsNow==NULL)	//hope that client will make a copy of the data
						gotSensorsNow = new StreamableDataContainer(bytesize);
					gotSensorsNow->setDataStream(readbuffer, bytesize);

					pthread_mutex_unlock(sensorsDemanded);
				}
				else //GetSensors
				{
					SimulationCommand acksens = queuedGetSensorCommands[0];

					acksens.setMessageACKGetSensors(readbuffer, bytesize);

					queuedGetSensorCommands.pop_front();

					sensorHandler->applyACKGetSensors(acksens);

					pthread_mutex_unlock(sensorLock);	//sensorhandler must not change until ACK is returned
				}
				break;
			}
			case StartSimulation:	//tells PID (TODO: and stuff?)
			{
				readDataToBuffer();
				pthread_mutex_lock(pidLock);
				PID = *((pid_t*) readbuffer);
				pthread_mutex_unlock(pidLock);
				break;
			}
			case StopSimulation:
			{
				if(shutDown!=true)
				{
					cout<<"Sim "<<simID<<" shut itself down!?\n";
				}
				pthread_mutex_unlock(deleteMeLock);
				//pthread_mutex_unlock(sensorsDemanded);	//Achtung: Möglicherweise werden falsche Sensordaten zurückgegeben, wenn die Simulation beendet, während ein GetSensorsNow läuft
				//cout<<"MAN: Simulation Sensor Thread Finished!\n";
				return; //will not read from pipe anymore
				break;
			}
			default:
			{
				cout<<"ERROR: Got unknown answer from Simulation!\n";
				break;
			}
			}
		}
	}

	void runQueueWriter()
	{
		while(1)
		{
			pthread_mutex_lock(queueLock);
	
			if(!commandQueue.empty())
			{
				SimulationCommand* cmd = &(commandQueue[0]);

				pthread_mutex_unlock(queueLock);

				MessageType mess = cmd->getMessageType();

				switch(mess)
				{
				case Step:
					step(cmd->getAdditionalInfo());
					break;
				case SetAction:
					setAction(cmd->getBody());
					break;
				case GetSensors:
					pthread_mutex_lock(sensorLock);
					queuedGetSensorCommands.push_back(*cmd);
					pthread_mutex_unlock(sensorLock);
					getSensors();
					break;
				case GetSensorsNow:
					getSensors();
					break;
				case StopSimulation:
					writeData((unsigned char*) &mess, sizeof(mess));
					flushWriteBuffer();
					//cout<<"MAN: Simulation QueueWriter Thread Finished!\n";
					return; //will not accept commands anymore
					break;
				case StartSimulation:
					writeData((unsigned char*) &mess, sizeof(mess));
					flushWriteBuffer();
					break;
				default:
					cout<<"ERROR: Unknown command type in queue!\n";
					break;
				}

				pthread_mutex_lock(queueLock);

				commandQueue.pop_front();

				if(commandQueue.empty())
					pthread_mutex_lock(emptyQueueLock);

				pthread_mutex_unlock(queueLock);
			}
			else	//avoid buffer underrun
			{
				pthread_mutex_unlock(queueLock);

				//set/reset "condition lock" for high responsibility
				pthread_mutex_lock(emptyQueueLock);
				pthread_mutex_unlock(emptyQueueLock);
			}
		}
	}

	static void* runQueueWriterThreadEntry(void* context)
	{
		((PipeServerSimulation*) context)->runQueueWriter();
		pthread_exit(NULL);
		return NULL;
	}

	static void* runSensorReaderThreadEntry(void* context)
	{
		((PipeServerSimulation*) context)->runSensorReader();
		pthread_exit(NULL);
		return NULL;
	}

protected:

	friend class SimulationManager;

	PipeServerSimulation(unsigned int set_simID, unsigned int set_buffersize, char* set_output_pipename, char* set_input_pipename, char* set_cmdline) : Simulation(0,NULL) , PipeRW(set_buffersize)
	{
		output_pipename=new char[strlen(set_output_pipename)+1];
		input_pipename=new char[strlen(set_input_pipename)+1];
		cmdline=new char[strlen(set_cmdline)+1];
		strncpy(output_pipename,set_output_pipename,strlen(set_output_pipename)+1);
		strncpy(input_pipename,set_input_pipename,strlen(set_input_pipename)+1);
		strncpy(cmdline,set_cmdline,strlen(set_cmdline)+1);

		sensorHandler=NULL;	//NULL until set

		simID=set_simID;

		PID=0;

		gotSensorsNow=NULL;

		shutDown=false;

		pidLock = new pthread_mutex_t;
		pthread_mutex_init(pidLock, NULL);

		sensorLock = new pthread_mutex_t;
		pthread_mutex_init(sensorLock, NULL);

		sensorsDemanded = new pthread_mutex_t;
		pthread_mutex_init(sensorsDemanded, NULL);
		pthread_mutex_lock(sensorsDemanded);

		deleteMeLock = new pthread_mutex_t;
		pthread_mutex_init(deleteMeLock, NULL);
		pthread_mutex_lock(deleteMeLock);


		queueLock = new pthread_mutex_t;
		pthread_mutex_init(queueLock, NULL);

		emptyQueueLock = new pthread_mutex_t;
		pthread_mutex_init(emptyQueueLock, NULL);
		pthread_mutex_lock(emptyQueueLock);



		/*We create a output-FIFO*/
		int output_fd=(mkfifo(output_pipename, O_RDWR));
		if(output_fd == -1)
		{
			fprintf(stderr, "Can't create output pipe. If it's already in use you should abort.\n");
		}

		//set really big maximum pipe size (see "cat /proc/sys/fs/pipe-max-size"):
		fcntl(output_fd, F_SETPIPE_SZ, 1048576);

		//Set permissions
		char cmd[128];
		snprintf(cmd,128,"chmod 700 %s", output_pipename);
		system(cmd);

		/*We open the fifo for writing*/
		if((output_fifo=fopen(output_pipename, "r+b"))==NULL)
		{
			fprintf(stderr, "Can't open the output pipe.....\n");
		}


		/*We create a input-FIFO*/
		int input_fd=(mkfifo(input_pipename, O_RDWR));
		if(input_fd == -1)
		{
			fprintf(stderr, "Can't create input pipe. If it's already in use you should abort.\n");
		}

		//set really big maximum pipe size (see "cat /proc/sys/fs/pipe-max-size"):
		fcntl(input_fd, F_SETPIPE_SZ, 1048576);

		//Set permissions
		snprintf(cmd,128,"chmod 700 %s", input_pipename);
		system(cmd);

		/*We open the fifo for reading*/
		if((input_fifo=fopen(input_pipename, "r+b"))==NULL)
		{
			fprintf(stderr, "Can't open the input pipe.....\n");
		}


		//start queuethread
		if(pthread_create(&queueThread,NULL,runQueueWriterThreadEntry,this)!=0)
			cout<<"ERROR: Can't create queue thread! Out of resources?\n";
		if(pthread_create(&sensorThread,NULL,runSensorReaderThreadEntry,this)!=0)
			cout<<"ERROR: Can't create sensor thread! Out of resources?\n";
	}

	~PipeServerSimulation()
	{
		pthread_cancel(queueThread);	//needed when thread did not exit...
		pthread_cancel(sensorThread);

		delete queueLock;
		delete sensorLock;
		delete sensorsDemanded;
		if(gotSensorsNow!=NULL)
			delete gotSensorsNow;

		//close pipes
		fclose(output_fifo);
		fclose(input_fifo);

		//delete pipes
		char cmd[80];
		snprintf(cmd,80,"rm -f %s", output_pipename);
		system(cmd);
		snprintf(cmd,80,"rm -f %s", input_pipename);
		system(cmd);

		delete[] output_pipename;
		delete[] input_pipename;
		delete[] cmdline;

		pthread_join(queueThread,NULL);		//important for resource management...
		pthread_join(sensorThread,NULL);
	}
	
};
