/**
 * All simulation-command related datatypes including the class SimulationCommand itself (Header+Implementation)
 * 
 * Copyright 2013 Fabian Schrodt, FSchrodt@gmx.de
 * 
 * This file is part of SimulationManager.
 * 
 * SimulationManager is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License version 3 as published by the Free Software Foundation.
 * 
 * SimulationManager is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with SimulationManager. If not, see http://www.gnu.org/licenses/.
 */

#pragma once

#include "SimulationData.h"
#include "SensorHandlerInterface.h"

#include <iostream>
#include <deque>
#include <string>
#include <cstring>
using namespace std;

/*
 * Auflistung aller Kommandos
 * */
enum MessageType
{
	/*
	 * ClientToServer
	 * */
	StartSimulation,
	StopSimulation,
	StopSimulationNow,
	Step,
	SetAction,
	GetSensors,
	GetSensorsNow,		//this is the only blocking command

	/*
	 * ServerToClient
	 * */
	ACK,
	ERR,
	ACKGetSensors
};

static const char* ErrorMessage[] =
{
	"MSG-ERROR: No command specified.",
	"MSG-ERROR: Simulation-ID is invalid.",
	"MSG-ERROR: Server is full at the moment. Please try again later.",
	"MSG-ERROR: Simulation process is unknown or insecure.",
	"MSG-ERROR: Error on closing Simulation.",
	"MSG-ERROR: You did not set any sensor-handler. Can not execute GetSensors (non-blocking), use GetSensorsNow instead (blocking).",
	"MSG-ERROR: Simulation has already stopped."
};

/*
 * Header containing all Information the manager might need to execute this command.
 * */

BEGIN_DATA
struct CommandHeader
{
	MessageType messageType;

	unsigned int simulationID;

	unsigned int messageID;

	/*
	 * "additionalInfo" may contain information used to assign an answer.
	 * For example, a pointer may be forwarded to the non-blocking command GetSensors which depicts the address at which the sensor values contained by the delayed answer ACKGetSensors may be written to.
	 * */
	unsigned int additionalInfo;

	unsigned int errorType;

	void print()
	{
		cout<<"\tmessageType = "<<messageType<<endl;
		cout<<"\tsimulationID = "<<simulationID<<endl;
		cout<<"\tmessageID = "<<messageID<<endl;
		cout<<"\tadditionalInfo = "<<additionalInfo<<endl;
		cout<<"\terrorType = "<<errorType<<endl;
	}
};
END_DATA

/*struct StreamableCommandHeader : public StreamableDataContainer
{
	CommandHeader* dat()
	{
		return (CommandHeader*) dataStream;
	}

	StreamableCommandHeader()
	{
		reconfigure(sizeof(CommandHeader));
	}
};*/

BEGIN_DATA
struct SimulationInfo
{
	char simName[];
};
END_DATA

/*struct StreamableSimulationInfo : public StreamableDataContainer
{
	SimulationInfo* dat()
	{
		return (SimulationInfo*) dataStream;
	}

	StreamableSimulationInfo(unsigned int n)
	{
		reconfigure(n);
	}
};*/



/*
 * Reusable SimulationCommand Type
 * */
class SimulationCommand
{
	friend class SimulationManager;
	friend class PipeServerSimulation;

	//MESSAGE HEADER.

	StreamableDataContainer header;	//always existent
	CommandHeader* headerData;

	//MESSAGE BODY / STREAMABLE DATA

	StreamableDataContainer body;	//may be appended. a copy of the given body will be created

	/*
	 * Private message type setters (the causing command should be used)
	 * */
	void setMessageACK(unsigned int set_additionalInfo)
	{
		headerData->messageType=(ACK);
		headerData->additionalInfo=(set_additionalInfo);

		body.clear();
	}

	void setMessageERR(unsigned int set_errorType)
	{
		cout<<"MAN: MESSAGE ERROR (msgID="<<header.dat<CommandHeader>()->messageID<<", simID="<<header.dat<CommandHeader>()->simulationID<<"): "<<ErrorMessage[set_errorType]<<endl;

		headerData->messageType=(ERR);
		headerData->errorType=(set_errorType);

		body.clear();
	}

	void setMessageACKGetSensors(void* set_stream, unsigned int set_byteSize)
	{
		headerData->messageType=(ACKGetSensors);

		body.setDataStream(set_stream, set_byteSize);
	}

public:

	void print()
	{
		cout<<"SimulationCommand: \n";
		cout<<"HEADER:\n";
		headerData->print();
		cout<<"BODY:\n";
		cout<<"\tData of size "<<body.getByteSize()<<endl;
		cout<<endl;
	}

	/*
	 * Getters (only) for streaming
	 * */
	const StreamableDataContainer* getHeader() const
	{
		return &header;
	}
	StreamableDataContainer* getHeader()  //DIRTY?
	{
		return &header;
	}

	const StreamableDataContainer* getBody() const
	{
		return &body;
	}
	StreamableDataContainer* getBody()	//DIRTY?
	{
		return &body;
	}

	/*
	 * Getters
	 * */
	MessageType getMessageType()
	{
		return header.dat<CommandHeader>()->messageType;
	}

	unsigned int getMessageID()
	{
		return header.dat<CommandHeader>()->messageID;
	}

	unsigned int getSimulationID()
	{
		return header.dat<CommandHeader>()->simulationID;
	}

	const char* getErrorMessage()
	{
		return ErrorMessage[header.dat<CommandHeader>()->errorType];
	}

	unsigned int getAdditionalInfo()
	{
		return header.dat<CommandHeader>()->additionalInfo;
	}

	/*
	 * Message type setters
	 * */
	void setMessageStartSimulation(const char* simulation_name)
	{
		headerData->messageType=(StartSimulation);

		unsigned int len = strlen(simulation_name);
		body.setDataStream((void*) simulation_name, len+1);
	}

	void setMessageStopSimulation(unsigned int set_simulationID)
	{
		headerData->messageType=(StopSimulation);
		headerData->simulationID=(set_simulationID);

		body.clear();
	}

	void setMessageStopSimulationNow(unsigned int set_simulationID, unsigned int set_additionalInfo = 0)
	{
		headerData->messageType=(StopSimulationNow);
		headerData->simulationID=set_simulationID;
		headerData->additionalInfo=set_additionalInfo;

		body.clear();
	}

	void setMessageStep(unsigned int set_simulationID, unsigned int nr_of_steps = 1)
	{
		headerData->messageType=(Step);
		headerData->simulationID=(set_simulationID);
		headerData->additionalInfo=(nr_of_steps);

		body.clear();
	}

	void setMessageSetAction(unsigned int set_simulationID, StreamableDataContainer* set_data)
	{
		headerData->messageType=(SetAction);
		headerData->simulationID=(set_simulationID);

		body.setDataStream(set_data->getDataStream(), set_data->getByteSize());
	}

	void setMessageGetSensors(unsigned int set_simulationID, unsigned int set_messageID, unsigned int set_additionalInfo = 0)
	{
		headerData->messageType=(GetSensors);
		headerData->messageID=(set_messageID);
		headerData->simulationID=(set_simulationID);
		headerData->additionalInfo=(set_additionalInfo);

		body.clear();
	}

	void setMessageGetSensorsNow(unsigned int set_simulationID)
	{
		headerData->messageType=(GetSensorsNow);
		headerData->simulationID=(set_simulationID);

		body.clear();
	}

	SimulationCommand& operator=(const SimulationCommand& set)
	{
		//cout<<"COMMAND COPY!\n";

		header.setDataStream(set.getHeader()->getDataStream(), set.getHeader()->getByteSize());
		body.setDataStream(set.getBody()->getDataStream(), set.getBody()->getByteSize());

		return *this;
	}

	// copy constructor for commands
	SimulationCommand(const SimulationCommand& set)
    {
		//cout<<"COMMAND COPY!\n";

		header.setDataStream(set.getHeader()->getDataStream(), set.getHeader()->getByteSize());
		body.setDataStream(set.getBody()->getDataStream(), set.getBody()->getByteSize());

		headerData = header.dat<CommandHeader>();
    }

	// stream copy constructor: constructs a SimulationCommand using the given header and body streams.
	SimulationCommand(void* header_stream, void* body_stream = NULL, unsigned int body_stream_size = 0)
    {
		header.setDataStream(header_stream, header.getByteSize());

		headerData = header.dat<CommandHeader>();

		if(body_stream_size!=0)
		{
			body.setDataStream(body_stream, body_stream_size);
		}
		else
			body.clear();
    }

	//Default-constructor. Use one of the above message setters after construction!
	SimulationCommand()
	{
		header.reconfigure(sizeof(CommandHeader));

		headerData = header.dat<CommandHeader>();

		headerData->messageType=ERR;
		headerData->errorType=(0);
	}

	~SimulationCommand()
	{
	}
};
